import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import {
  createQueryBuilder,
  updateQueryBuilder,
  whereCondition,
  makeJoins,
  countQueryCondition,
  searchConditionRecord,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  decodeAndParseFields,
  encodeAndStringifyFields,
  insertActivityLog,
  uniqueIdGenerator,
} from "../helper/general.js";
import RequirementAnalysis from "../sequelize/BcpRequirementAnalysisSchema.js";

export const createUpdateBcpRequirementAnalysis = async (req, res) => {
  const { id, department } = req.body;

  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }

  req.body = await encodeAndStringifyFields(req.body);

  if(!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "BCPRA",
      "bcp_requirement_analysis",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }


  const { query, values } = id
    ? updateQueryBuilder(RequirementAnalysis, req.body)
    : createQueryBuilder(RequirementAnalysis, req.body);
  await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Requirement Analysis",
    `This user ${status} a new Requirement Analysis Record`
  );
  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getBcpRequirementAnalysis = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "bcp_requirement_analysis",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "bcp_requirement_analysis.name",
    "bcp_requirement_analysis.description",
    "bcp_requirement_analysis.unique_id",
    "organization.name",  
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "location",
      onCondition:
        "location.id = bcp_requirement_analysis.location",
    },
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = bcp_requirement_analysis.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = bcp_requirement_analysis.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = bcp_requirement_analysis.department",
    },
    {
      type: "left",
      targetTable: "users as lead_assessor_users",
      onCondition:
        "lead_assessor_users.id = bcp_requirement_analysis.lead_assessor",
    },
    {
      type: "left",
      targetTable: "roles as lead_assessor_roles",
      onCondition: "lead_assessor_roles.id = lead_assessor_users.role",
    },
    {
      type: "left",
      targetTable: "department as lead_assessor_department",
      onCondition:
        "lead_assessor_department.id = lead_assessor_users.department",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const bcpRequirementAnalysisQuery = `SELECT bcp_requirement_analysis.*,
  CONCAT(lead_assessor_users.name , ' ' , lead_assessor_users.surname) AS lead_assessor_name , lead_assessor_users.profile as lead_assessor_profile, lead_assessor_roles.name as lead_assessor_role , lead_assessor_department.name as lead_assessor_department, organization.name as organization_name , department.name as department_name , location.name as location_name, users.name as created_by_name , users.surname AS created_by_surname , users.profile AS created_by_profile FROM bcp_requirement_analysis ${joinCondition} WHERE bcp_requirement_analysis.deleted = 0 ${searchCondition} ${condition}`;
  
  let [requirementAnalysis] = await db.query(bcpRequirementAnalysisQuery);
  requirementAnalysis = await decodeAndParseFields(requirementAnalysis);

  for (let item of requirementAnalysis) {
    let assessment_participants = item?.assessment_participants;
    assessment_participants = assessment_participants?.map((e) => e?.user_id)||[];
    // employee query
    const employeeQuery = `SELECT users.id as user_id, CONCAT(users.name , ' ' , users.surname) AS name,
        users.profile, roles.name AS role, department.name AS department FROM users LEFT JOIN roles ON roles.id = users.role LEFT JOIN department ON department.id = users.department WHERE users.id IN (${assessment_participants})`;
    const [employees] = await db.query(employeeQuery);
    item.assessment_participants = employees;
  }

  const totalRecord = await countQueryCondition(bcpRequirementAnalysisQuery);

  return sendResponse(res, 200, requirementAnalysis, totalRecord);
};

export const deleteBcpRequirementAnalysis = async (req, res) => {
  const { id } = req.params;
  await deleteRecord("bcp_requirement_analysis", id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "bcp_requirement_analysis",
    `This user deleted a new bcp_requirement_analysis Record for organization ${id}`
  );
  return sendResponse(res, 200, "Record deleted successfully");
};
