import asyncHandler from "express-async-handler";
import db from "../db-config.js";
import {
  countQueryCondition,
  decodeSingle_statement,
  deleteSettingRecord,
  encodeSingle_statement,
  insertActivityLog,
  makeJoins,
  whereCondition,
  settingsUpdater,
  storeError,
  createQueryBuilder,
  searchConditionRecord,
  uniqueIdGenerator,
} from "../helper/general.js";
import CategoryType from "../sequelize/CategoryTypeSchema.js";
import { sendResponse } from "../helper/wrapper.js";

/**Function to create new category Type for employee  */
export const createUpdateCategoryType = async (req, res) => {
  const { id, name, description, organization } = req.body;
  const encodedDescription = await encodeSingle_statement(description);
  req.body.description = encodedDescription;

  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    if (organization.length === 0) {
      return sendResponse(
        res,
        400,
        "Please Select at least one business Structure"
      );
    }
    const result = await settingsUpdater(
      CategoryType,
      organization,
      req.body,
      req.user
    );

    if (!result) {
      return sendResponse(res, 404, "No data found");
    }

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "Category Type",
      `This user Update Category Type which id is ${id}`
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    /** Insert new record */
    const insertPromises = organization.map(async (element) => {
      // Check if the category type already exists for the organization
      const [checkNameWithOrganization] = await db.query(
        `SELECT name FROM category_type WHERE name = ? AND deleted = "0" AND organization = ?`,
        [name, element]
      );

      // If record not found for that name with particular organization, insert the record
      if (checkNameWithOrganization.length === 0) {
        const unique_id = await uniqueIdGenerator(
          element,
          req?.body?.department || null,
          "CategoryType",
          "category_type",
          "unique_id",
          "unique_id"
        );
        req.body.unique_id = unique_id;

        req.body.created_by = req.user.sessionid;
        req.body.organization = element;
        // Insert Category Type details
        const { query, values } = createQueryBuilder(CategoryType, {
          ...req.body,
          organization: element,
        });
        await db.query(query, values);

        // Insert record for activity log
        await insertActivityLog(
          req.user.sessionid,
          "create",
          "Category Type",
          `This user created a new category '${name}' for this organization ${element}`
        );
      }
    });

    await Promise.all(insertPromises);

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single category Type */
export const viewCategoryType = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "category_type",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "category_type.name",
    "category_type.description",
    "organization.name",
    "CONCAT(createdUser.name, ' ', createdUser.surname)",
  ];

  /** If value comes with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "createdUser.id = category_type.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = category_type.organization",
    },
  ];
  const joinsRecord = await makeJoins(joins);

  const categoryTypeQuery = `SELECT category_type.*, organization.name as organization_name ,createdUser.name as created_by , createdUser.surname as created_by_surname , createdUser.id as created_by_id , createdUser.profile as created_by_profile FROM category_type ${joinsRecord} where category_type.deleted = 0 ${searchCondition} ${condition}`;
  const [categoryType] = await db.query(categoryTypeQuery);

  for (const item of categoryType) {
    item.description = await decodeSingle_statement(item.description);
    if (req.query.grouped == "true") {
      const [organizations] = await db.query(
        `SELECT organization FROM category_type WHERE deleted = 0 AND name = ?`,
        [item.name]
      );
      const arr = organizations.map((item) => item.organization);
      item.organizations = arr;
    }
  }

  /**Count all organization */
  const totalRecord = await countQueryCondition(categoryTypeQuery);

  return sendResponse(res, 200, categoryType, totalRecord);
};

/**Function to delete a specific category Type */
export const deleteCategoryType = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteSettingRecord("category_type", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "delete", "category_type", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

/**function to get category_type according to organization */
// export const getCategoryAccordingToOrg = asyncHandler(async (req, res) => {
//   try {
//     const { organization } = req.params;
//     const Query = `SELECT id,name,description,organization
//     FROM category_type where deleted = 0 AND organization = ${organization} ORDER BY id DESC`;
//     const Result = await db.query(Query);

//     return res.status(200).json({ status: true, data: Result });
//   } catch (error) {
//     // Check if an error occurs, then send that error to the log file
//     storeError(error);
//     return res.status(500).json({ status: false, message: error.message });
//   }
// });
