import db from "../db-config.js";
import {
  makeJoins,
  whereCondition,
  searchConditionRecord,
  getOrganizationAccordingToDepartment,
  updateQueryBuilder,
  createQueryBuilder,
  encodeAndStringifyFields,
  insertActivityLog,
  decodeAndParseFields,
  deleteRecord,
  countQueryCondition,
  uniqueIdGenerator,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import ContractorNCR from "../sequelize/ContractorNCRRecordingSchema.js";

export const createUpdateContractorNCR = async (req, res) => {
  const { id, department } = req.body;
  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;

  if (!id) {
    const unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "ContractorNCR",
      "contractor_ncr_recording",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
  }

  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(ContractorNCR, req.body)
    : createQueryBuilder(ContractorNCR, req.body);
  const result = await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "Contractor NCR",
    id ? id : result[0].insertId
  );
  return sendResponse(res, 200, `Record ${status} Successfully`);
};

export const getContractorNCR = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "contractor_ncr_recording",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "contractor_ncr_recording.raised_by_employee",
    "contractor_ncr_recording.ncr_description",
    "contractor_ncr_recording.type_of_work",
    "contractor_ncr_recording.immediate_action_taken",
    "contractor_ncr_recording.legal_non_compliance",
    "contractor_ncr_recording.system_non_conformance",
    "contractor_ncr_recording.customer_non_compliance",
    "contractor_ncr_recording.name_of_legislation",
    "contractor_ncr_recording.customer_name",
    "contractor_ncr_recording.provision_requirement",
    "contractor_ncr_recording.standard_clause_name",
    "contractor_ncr_recording.contravened_contract_clause",
    "contractor_ncr_recording.ncr_categories",
    "contractor_ncr_recording.classification",
    "contractor_ncr_recording.name_of_site",
    "contractor_ncr_recording.name_of_external_raised_by",
    "contractor_ncr_recording.contact_details",
    "contractor_ncr_recording.root_causes",
    "contractor_ncr_recording.corrective_action_title",
    "contractor_ncr_recording.corrective_action_description",
    "contractor_ncr_recording.corrective_action_progress",
    "contractor_ncr_recording.corrective_status",
    "contractor_ncr_recording.preventative_action_title",
    "contractor_ncr_recording.preventative_action_description",
    "contractor_ncr_recording.preventative_action_progress",
    "contractor_ncr_recording.preventative_status",
    "contractor_ncr_recording.human_resources",
    "contractor_ncr_recording.technological_resources",
    "contractor_ncr_recording.type_of_resource",
    "contractor_ncr_recording.amount",
    "contractor_ncr_recording.additional_comments",
    "CONCAT(createdUser.name , ' ' createdUser.surname)",
    "organization.name",
    "department.name",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = contractor_ncr_recording.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = contractor_ncr_recording.organization",
    },
    {
      type: "left",
      targetTable: "organization AS internalRaisedOrg",
      onCondition:
        "internalRaisedOrg.id = contractor_ncr_recording.internal_raised_business",
    },
    {
      type: "left",
      targetTable: "organization AS internalAgainstOrganization",
      onCondition:
        "internalAgainstOrganization.id = contractor_ncr_recording.internal_raised_against_business",
    },
    {
      type: "left",
      targetTable: "department AS internalAgainstDepartment",
      onCondition:
        "internalAgainstDepartment.id = contractor_ncr_recording.internal_raised_against_department",
    },
    {
      type: "left",
      targetTable: "users AS internalResponsible",
      onCondition:
        "internalResponsible.id = contractor_ncr_recording.internal_responsible_person",
    },
    {
      type: "left",
      targetTable: "users AS correctiveResponsible",
      onCondition:
        "correctiveResponsible.id = contractor_ncr_recording.corrective_responsible_person",
    },
    {
      type: "left",
      targetTable: "department AS correctiveDepartment",
      onCondition:
        "correctiveDepartment.id = contractor_ncr_recording.corrective_responsible_department",
    },
    {
      type: "left",
      targetTable: "department AS preventiveDepartment",
      onCondition:
        "preventiveDepartment.id = contractor_ncr_recording.preventative_responsible_department",
    },
    {
      type: "left",
      targetTable: "users AS preventiveUser",
      onCondition:
        "preventiveUser.id = contractor_ncr_recording.preventative_responsible_person",
    },
    {
      type: "left",
      targetTable: "name_of_standard",
      onCondition:
        "name_of_standard.id = contractor_ncr_recording.name_of_standard",
    },
    {
      type: "left",
      targetTable: "contractor_registration",
      onCondition:
        "contractor_registration.id = contractor_ncr_recording.contractor",
    },
    {
      type: "left",
      targetTable: "project_registration",
      onCondition:
        "project_registration.id  = contractor_ncr_recording.project",
    },
    {
      type: "left",
      targetTable: "repository as  stop_certification",
      onCondition:
        "stop_certification.id = contractor_ncr_recording.stop_certificate_ddrm_id",
    },
    {
      type: "left",
      targetTable: "repository as ncr_report",
      onCondition:
        "ncr_report.id = contractor_ncr_recording.ncr_report_ddrm_id",
    },
    {
      type: "left",
      targetTable: "users as recipient_user",
      onCondition:
        "recipient_user.id = contractor_ncr_recording.recipient_user_id",
    },
    {
      type: "left",
      targetTable: "users as issuer_user",
      onCondition: "issuer_user.id = contractor_ncr_recording.issuer_user_id",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = contractor_ncr_recording.internal_location",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const contractorNCRQuery = `SELECT contractor_ncr_recording.* ,
  location.name as incident_location_name, stop_certification.url as stop_certificate_file, 
  ncr_report.url as ncr_report_file, CONCAT(recipient_user.name , ' ' , recipient_user.surname) AS recipient_user_name,
  recipient_user.profile AS recipient_user_profile, CONCAT(issuer_user.name , ' ' , issuer_user.surname) AS issuer_user_name, 
  issuer_user.profile AS issuer_user_profile, users.name as created_by_name,users.surname as created_by_surname, 
  users.profile as created_by_profile ,organization.name as organization_name , 
  internalRaisedOrg.name AS internal_raised_business_name , internalAgainstOrganization.name AS internal_raised_against_business_name , 
  internalAgainstDepartment.name AS internal_raised_against_department_name , 
  CONCAT(internalResponsible.name , ' ' , internalResponsible.surname) AS internal_responsible_person_name , 
  internalResponsible.unique_id AS internal_employee_no , internalResponsible.profile AS internal_responsible_person_profile , 
  CONCAT(correctiveResponsible.name , ' ' , correctiveResponsible.surname) as corrective_responsible_person_name , 
  correctiveDepartment.name AS corrective_responsible_department_name , 
  preventiveDepartment.name AS preventative_responsible_department_name ,
  CONCAT(preventiveUser.name , ' ' , preventiveUser.surname) AS preventative_responsible_person_name  , 
  name_of_standard.name AS name_of_standard_name , contractor_registration.contractor_name AS contractor_name , 
  contractor_registration.cipc_registration_number AS contractor_cipc_registration_number , 
  project_registration.project_title AS project_name  
  FROM contractor_ncr_recording ${joinCondition} 
  WHERE contractor_ncr_recording.deleted = 0 ${searchCondition} ${condition}`;
  // console.log(ncrRecordingQuery);
  let [contractorNCR] = await db.query(contractorNCRQuery);
  contractorNCR = await decodeAndParseFields(contractorNCR);
  for (const i of contractorNCR) {
    const raisedBy = i.raised_by;
    let [projectMembers] = await db.query(
      `SELECT project_members FROM project_registration WHERE id = ${i.project}`
    );
    projectMembers = JSON.parse(projectMembers[0]?.project_members);

    i.raised_by_details = [projectMembers.filter((x) => x.id == raisedBy)][0];

    if (i.raised_type == 0) {
      const [user] = await db.query(
        `SELECT CONCAT(name , ' ' , surname ) AS raised_by_name , users.phone AS raised_by_contact  FROM users WHERE id = ${i.internal_responsible_person}`
      );
      i.internal_responsible_person_name = user[0]?.raised_by_name;
      i.internal_responsible_person_contact = user[0]?.raised_by_contact;
    } else {
      const [contractor] = await db.query(
        `SELECT contractor_name , phone_number FROM contractor_registration WHERE id = ${i.external_raised_person}`
      );
      i.external_raised_person_name = contractor[0]?.contractor_name;
      i.external_raised_person_contact = contractor[0]?.phone_number;
    }
    const categories = i.ncr_categories;
    let cat = [];
    for (const category of categories) {
      const [name] = await db.query(
        `SELECT name FROM incident_category WHERE id = ${category}`
      );
      cat.push(name[0]?.name);
    }
    i.ncr_categories_name = cat;
    const requirement = i?.human_resource_requirements;
    if (requirement) {
      for (const r of requirement) {
        if (r.role) {
          const [name] = await db.query(
            `SELECT name FROM roles WHERE id = ${r.role}`
          );
          r.role_name = name[0].name;
        }
      }
    }
  }

  const totalRecord = await countQueryCondition(contractorNCRQuery);

  return sendResponse(res, 200, contractorNCR, totalRecord);
};

export const deleteContractorNcrRecording = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(ContractorNCR, id);
  await insertActivityLog(req.user.sessionid, "delete", "Contractor NCR", id);
  return sendResponse(res, 200, "Record deleted successfully");
};

export const uploadSignatureContractorNCR = async (req, res) => {
  // const { sidebar_id, id } = req.body;
  // const ddrm_id = await uploadToDDRM(sidebar_id, signature, req);

  const { query, values } = updateQueryBuilder(ContractorNCR, {
    ...req.body,
  });
  const [result] = await db.query(query, values);
  if (result.affectedRows > 0) {
    return sendResponse(res, 200, "certificate uploaded successfully");
  } else {
    return sendResponse(res, 400, "Error uploading signature");
  }
};
