import db from "../db-config.js";
import { getCalendar } from "../helper/calendar.js";
import {
  checkCircularDependency,
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  updateQueryBuilder,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import TaskTemplateList from "../sequelize/TaskTemplateListSchema.js";
import CustomTaskCreation from "../sequelize/CustomTaskCreationSchema.js";

function sortTasks(tasks) {
  const graph = new Map();
  const reverseGraph = new Map();

  // Build the graph and reverse graph
  tasks.forEach((task) => {
    graph.set(task.id, new Set());
    reverseGraph.set(task.id, new Set());
  });

  tasks.forEach((task) => {
    const successors = Array.isArray(task.successor_task_template)
      ? task.successor_task_template
      : [task.successor_task_template].filter(Boolean);
    const predecessors = Array.isArray(task.predecessor_task_template)
      ? task.predecessor_task_template
      : [task.predecessor_task_template].filter(Boolean);

    successors.forEach((succId) => {
      if (graph.has(succId)) {
        graph.get(task.id).add(succId);
        reverseGraph.get(succId).add(task.id);
      }
    });

    predecessors.forEach((predId) => {
      if (graph.has(predId)) {
        graph.get(predId).add(task.id);
        reverseGraph.get(task.id).add(predId);
      }
    });
  });

  function dfs(node, visited, stack, graph) {
    visited.add(node);
    for (let neighbor of graph.get(node)) {
      if (!visited.has(neighbor)) {
        dfs(neighbor, visited, stack, graph);
      }
    }
    stack.push(node);
  }

  function topologicalSort(graph) {
    const visited = new Set();
    const stack = [];
    for (let node of graph.keys()) {
      if (!visited.has(node)) {
        dfs(node, visited, stack, graph);
      }
    }
    return stack.reverse();
  }

  // Perform topological sort on both graphs
  const forwardOrder = topologicalSort(graph);
  const backwardOrder = topologicalSort(reverseGraph);

  // Combine both orders
  const scoreMap = new Map();
  forwardOrder.forEach((id, index) => {
    scoreMap.set(id, index);
  });
  backwardOrder.forEach((id, index) => {
    scoreMap.set(id, (scoreMap.get(id) || 0) + index);
  });

  // Final sorting based on combined scores
  const sortedIds = [...scoreMap.entries()]
    .sort((a, b) => a[1] - b[1])
    .map((entry) => entry[0]);

  // Map the sorted IDs back to the original task objects
  const taskMap = new Map(tasks.map((task) => [task.id, task]));
  return sortedIds.map((id) => taskMap.get(id));
}

async function findValidTasks(taskId, selectedType, searchType, allTasks) {
  const validTasks = [];
  const task = allTasks.find((t) => t.id === taskId);

  if (!task) {
    return validTasks;
  }

  for (const potentialTask of allTasks) {
    if (potentialTask.id === taskId) continue;

    let isValid = true;

    if (selectedType === "successor" && searchType === "predecessor") {
      // Check if adding potentialTask as predecessor would create a cycle
      isValid = !(await checkCycleInChain(potentialTask.id, taskId, allTasks));
    } else if (selectedType === "predecessor" && searchType === "successor") {
      // Check if adding potentialTask as successor would create a cycle
      isValid = !(await checkCycleInChain(taskId, potentialTask.id, allTasks));
    }

    if (isValid) {
      validTasks.push(potentialTask);
    }
  }

  return validTasks;
}

async function checkCycleInChain(start, end, allTasks) {
  const visited = new Set();

  function dfs(currentId) {
    if (currentId === end) return true;
    if (visited.has(currentId)) return false;

    visited.add(currentId);

    const currentTask = allTasks.find((t) => t.id === currentId);
    if (currentTask) {
      if (
        currentTask.successor_task_template &&
        dfs(currentTask.successor_task_template)
      ) {
        return true;
      }
    }

    return false;
  }

  return dfs(start);
}

function fetchLinkedTasks(actionTemplateList, selectedTaskId) {
  const taskMap = new Map();

  actionTemplateList.forEach((task) => {
    taskMap.set(task.id, task);
  });

  const result = new Set();

  function dfs(taskId) {
    if (!taskMap.has(taskId) || result.has(taskId)) return;

    const task = taskMap.get(taskId);
    result.add(taskId); // Add the task to the result set

    if (task.successor_task_template) {
      dfs(task.successor_task_template);
    }

    if (task.predecessor_task_template) {
      dfs(task.predecessor_task_template);
    }
  }

  dfs(selectedTaskId);

  return Array.from(result).map((id) => taskMap.get(id));
}

export const getValidTasks = async (req, res) => {
  const { task_id, selected_type, organization } = req.query;

  if (
    !selected_type ||
    (selected_type !== "successor" && selected_type !== "predecessor")
  ) {
    return sendResponse(
      res,
      400,
      "Invalid selected_type. Must be 'successor' or 'predecessor'."
    );
  }

  try {
    const [allTasks] = await db.query(
      `SELECT id, predecessor_task_template, successor_task_template 
       FROM task_template_list 
       WHERE organization = ?`,
      [organization]
    );

    const searchType =
      selected_type === "successor" ? "predecessor" : "successor";
    const validTasks = await findValidTasks(
      task_id,
      selected_type,
      searchType,
      allTasks
    );

    if (validTasks.length === 0) {
      return sendResponse(res, 200, "No valid tasks found", []);
    }

    const validTaskIds = validTasks.map((task) => task.id);
    const [fullValidTasks] = await db.query(
      `SELECT * FROM task_template_list WHERE id IN (?)`,
      [validTaskIds]
    );

    return sendResponse(
      res,
      200,
      "Valid tasks retrieved successfully",
      fullValidTasks
    );
  } catch (error) {
    console.error("Error in getValidTasks:", error);
    return sendResponse(
      res,
      500,
      "An error occurred while retrieving valid tasks"
    );
  }
};

/** Function to create and update Custom Task Creation */
export const createUpdateCustomTaskCreation = async (req, res) => {
  let { id, task_data, department } = req.body;
  // console.log(req.body);

  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }
  if (
    task_data.predecessor_task_template ||
    task_data.successor_task_template
  ) {
    const circularDependency = await checkCircularDependency(
      id, // This might be undefined for new records
      task_data.predecessor_task_template,
      task_data.successor_task_template,
      "custom_task_creation"
    );
    if (circularDependency) {
      return sendResponse(
        res,
        400,
        "Circular dependency detected. Cannot create or update the record."
      );
    }
  }
  let status = id ? "Updated" : "Created";

  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  let attachment = [];
  if (req.files) {
    for (let i = 0; req.files[`task_data[${i}][file]`] != undefined; i++) {
      {
        const file = req.files[`task_data[${i}][file]`];
        if (file) {
          attachment.push(await uploadFile("custom_task_creation", file));
        } else {
          attachment.push(req.body[`task_data[${i}][file]`]);
        }
      }
    }
  }

  if (id != undefined) {
    const [result] = await db.query(
      `SELECT * FROM custom_task_creation WHERE id = ${id}`
    );
    await db.query(
      `UPDATE custom_task_creation SET predecessor_task_template = ${result[0]?.predecessor_task_template} WHERE id = ${result[0]?.successor_task_template}`
    );
    await db.query(
      `UPDATE custom_task_creation SET successor_task_template = ${result[0]?.successor_task_template} WHERE id = ${result[0]?.predecessor_task_template}`
    );
  }

  task_data = JSON.parse(task_data);

  const arr = [];
  let i = 0;
  for (let action of task_data) {
    action.id = req.body.id;
    action.template = req.body.template;
    action.organization = req.body.organization;
    action.department = req.body.department;
    action[id ? "updated_by" : "created_by"] = req.user.sessionid;
    action.attachment = attachment[i] ? attachment[i++] : null;
    const startDate = new Date(action.start_date);
    const endDate = new Date(action.end_date);

    const differenceInMilliseconds = endDate - startDate;

    const differenceInDays = differenceInMilliseconds / (1000 * 60 * 60 * 24);

    const totalWorkingHours = differenceInDays * 8;
    action.estimated_time = totalWorkingHours;
    action = await encodeAndStringifyFields(action);
    const { query, values } = id
      ? req.body.save_type == "template"
        ? updateQueryBuilder(TaskTemplateList, action)
        : updateQueryBuilder(CustomTaskCreation, action)
      : req.body.save_type == "template"
      ? createQueryBuilder(TaskTemplateList, action)
      : createQueryBuilder(CustomTaskCreation, action);
    console.log(query);
    const [result] = await db.query(query, values);
    if (action.predecessor_task_template) {
      let predId = action.predecessor_task_template;
      await db.query(
        `UPDATE custom_task_creation SET successor_task_template = ${
          id ? id : result.insertId
        } WHERE id = ${predId}`
      );
    }
    if (action.successor_task_template) {
      let sucId = action.successor_task_template;
      await db.query(
        `UPDATE custom_task_creation SET predecessor_task_template = ${
          id ? id : result.insertId
        } WHERE id = ${sucId}`
      );
    }
    arr.push({ name: action?.name, id: id ? id : result.insertId });
  }

  /** Insert record for activity log */
  await insertActivityLog(
    req.user.sessionid,
    id ? "update" : "create",
    "Custom Task Creation",
    `This user ${req.user.sessionid} has ${status} a new custom action`
  );

  return sendResponse(res, 200, `Record ${status} successfully`);
};

/** Function to get Custom Task Creation list */
export const getCustomTaskCreation = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "custom_task_creation",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const searchTableName = [
    "custom_task_creation.task_title",
    "custom_task_creation.task_description",
    "custom_task_creation.start_date",
    "custom_task_creation.end_date",
    "custom_task_creation.priority_level",
    "custom_task_creation.status",
    "custom_task_creation.task_budget",
    "custom_task_creation.task_resources",
    "organization.name",
    "department.name",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "custom_task_creation.created_by = createdUser.id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "custom_task_creation.organization = organization.id",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = custom_task_creation.updated_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "custom_task_creation.department = department.id",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = custom_task_creation.responsible_person",
    },
    {
      type: "left",
      targetTable: "roles",
      onCondition: "custom_task_creation.role = roles.id",
    },
    {
      type: "left",
      targetTable: "skills",
      onCondition: "custom_task_creation.skill = skills.id",
    },
    {
      type: "left",
      targetTable: "labels",
      onCondition: "custom_task_creation.label = labels.id",
    },
    {
      type: "left",
      targetTable: "certificates",
      onCondition: "custom_task_creation.certificate = certificates.id",
    },
    {
      type: "left",
      targetTable: "task_template_list",
      onCondition: "custom_task_creation.template = task_template_list.id",
    },
    {
      type: "left",
      targetTable: "category",
      onCondition: "custom_task_creation.tag = category.id",
    },
    {
      type: "left",
      targetTable: "custom_task_creation AS suc",
      onCondition: "custom_task_creation.successor_task_template = suc.id",
    },
    {
      type: "left",
      targetTable: "custom_task_creation AS pred",
      onCondition: "custom_task_creation.predecessor_task_template = pred.id",
    },
    {
      type: "left",
      targetTable: "sidebar AS moduleTable",
      onCondition: "moduleTable.id = custom_task_creation.module",
    },
    {
      type: "left",
      targetTable: "sidebar AS subModuleTable",
      onCondition: "subModuleTable.id = custom_task_creation.sub_module",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const customActionCreationQuery = `SELECT custom_task_creation.* , task_template_list.name AS template_name    , createdUser.name AS created_by_name , createdUser.surname AS created_by_surname, createdUser.profile AS created_by_profile , organization.name AS organization_name , department.name AS department_name , CONCAT(u1.name , ' ' , u1.surname) AS responsible_person_name , u1.profile AS responsible_person_profile , CONCAT(u2.name , ' ' ,u2.surname ) AS updated_by_name , u2.profile AS updated_by_profile , roles.name AS role_name , skills.name AS skill_name , labels.name AS label_name , certificates.name AS certificate_name , category.name AS tag_name , pred.task_title AS predecessor_task_template_name , suc.task_title AS successor_task_template_name , moduleTable.title AS module_name , subModuleTable.title AS sub_module_name FROM custom_task_creation ${joinCondition} WHERE custom_task_creation.deleted = 0 ${searchCondition}  ${condition}`;
  // console.log(customActionCreationQuery);
  let [customActionCreation] = await db.query(customActionCreationQuery);
  customActionCreation = await decodeAndParseFields(customActionCreation);
  for (const element of customActionCreation) {
    const ids = element.collaborators;
    if (element.human_resource_requirements) {
      for (const resource of element.human_resource_requirements) {
        const [roleN] = await db.query(
          `SELECT name FROM roles WHERE id = '${resource.role}'`
        );
        resource.role_name = roleN[0]?.name;
      }
    }
    let arr = [];
    for (const id of ids) {
      const [result] = await db.query(
        `SELECT CONCAT(name , ' ', surname) AS name , id , profile FROM users WHERE id = '${id}'`
      );
      arr.push({
        name: result[0]?.name,
        id: result[0]?.id,
        profile: result[0]?.profile,
      });
    }

    element.collaborator_details = arr;
  }
  const totalRecord = await countQueryCondition(customActionCreationQuery);
  if (req.query.type == `dependency`)
    customActionCreation = sortTasks(customActionCreation);

  return sendResponse(res, 200, customActionCreation, totalRecord);
};

export const deleteCustomTaskCreation = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(CustomTaskCreation, id);
  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Custom Task Creation",
    id
  );
  return sendResponse(res, 200, "Record deleted successfully");
};

export const fetchLinkedTemplates = async (req, res) => {
  let { organization, template } = req.query;
  let [result] = await db.query(
    `SELECT * FROM task_template_list WHERE organization = '${organization}'`
  );
  result = await decodeAndParseFields(result);
  for (let r of result) {
    r.human_resource_requirements = [{ role: "", no_of_peoples: "" }];
    r.technological_resources = [""];
  }
  template = Number(template);

  if (req.query.type == "true") {
    result = fetchLinkedTasks(result, template);
  } else {
    if (template) {
      [result] = await db.query(
        `SELECT * FROM task_template_list WHERE id = '${template}'`
      );
    }
    result = await decodeAndParseFields(result);
    for (let r of result) {
      r.human_resource_requirements = [{ role: "", no_of_peoples: "" }];
      r.technological_resources = [""];
    }
  }

  return sendResponse(res, 200, result);
};

export const fetchGanttChart = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "custom_task_creation",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    id,
    grouped: req.query.grouped,
    user: req.user,
    filter: req.query.filter,
  });
  const searchTableName = [
    "custom_task_creation.task_title",
    "custom_task_creation.task_description",
    "custom_task_creation.start_date",
    "custom_task_creation.end_date",
    "custom_task_creation.priority_level",
    "custom_task_creation.status",
    "custom_task_creation.task_budget",
    "custom_task_creation.task_resources",
    "organization.name",
    "department.name",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "custom_task_creation.created_by = createdUser.id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "custom_task_creation.organization = organization.id",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = custom_task_creation.updated_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "custom_task_creation.department = department.id",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = custom_task_creation.responsible_person",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const customActionCreationQuery = `SELECT custom_task_creation.* , createdUser.name AS created_by_name , createdUser.surname AS created_by_surname, createdUser.profile AS created_by_profile , organization.name AS organization_name , department.name AS department_name , CONCAT(u1.name , ' ' , u1.surname) AS assigned_to_name , u1.profile AS assigned_to_profile , CONCAT(u2.name , ' ' ,u2.surname ) AS updated_by_name , u2.profile AS updated_by_profile FROM custom_task_creation ${joinCondition} WHERE custom_task_creation.deleted = 0 ${searchCondition}  ${condition}`;
  let [customActionCreation] = await db.query(customActionCreationQuery);
  for (let i = 0; i < customActionCreation.length; i++) {
    const element = customActionCreation[i];
    element.start = element?.start_date;
    element.end = element?.end_date;
    element.type = "task";
    element.dependencies = element?.predecessor_task_template
      ? [element?.predecessor_task_template]
      : [];
    element.progress = 25;
    element.name = element?.task_title;
  }
  customActionCreation = await decodeAndParseFields(customActionCreation);
  customActionCreation = sortTasks(customActionCreation);
  // customActionCreation = customActionCreation.reverse();
  const totalRecord = await countQueryCondition(customActionCreationQuery);
  return sendResponse(res, 200, customActionCreation, totalRecord);
};

export const getCustomTaskCalender = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "custom_task_creation",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
    filter: req.query.filter,
  });
  const searchTableName = [
    "custom_task_creation.task_title",
    "custom_task_creation.task_description",
    "custom_task_creation.start_date",
    "custom_task_creation.end_date",
    "custom_task_creation.priority_level",
    "custom_task_creation.status",
    "custom_task_creation.task_budget",
    "custom_task_creation.task_resources",
    "organization.name",
    "department.name",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users as createdUser",
      onCondition: "custom_task_creation.created_by = createdUser.id",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "custom_task_creation.organization = organization.id",
    },
    {
      type: "left",
      targetTable: "users as u2",
      onCondition: "u2.id = custom_task_creation.updated_by",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "custom_task_creation.department = department.id",
    },
    {
      type: "left",
      targetTable: "users as u1",
      onCondition: "u1.id = custom_task_creation.responsible_person",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const customActionCreationQuery = `SELECT custom_task_creation.*  ,  createdUser.name AS created_by_name , createdUser.surname AS created_by_surname, createdUser.profile AS created_by_profile , organization.name AS organization_name , department.name AS department_name , CONCAT(u1.name , ' ' , u1.name) AS assigned_to_name , u1.profile AS assigned_to_profile , CONCAT(u2.name , ' ' ,u2.surname ) AS updated_by_name , u2.profile AS updated_by_profile FROM custom_task_creation ${joinCondition} WHERE custom_task_creation.deleted = 0 ${searchCondition}  ${condition}`;
  let [customActionCreation] = await db.query(customActionCreationQuery);
  customActionCreation = await decodeAndParseFields(customActionCreation);
  const templateStructure = {
    start: "start_date",
    end: "end_date",
    summary: "task_title",
  };
  const calenderPath = await getCalendar(
    customActionCreation,
    templateStructure,
    "custom_task_creation"
  );
  const totalRecord = await countQueryCondition(customActionCreationQuery);
  return sendResponse(res, 200, calenderPath, totalRecord);
};
export const fetchEmployeeAccordingToGivenData = async (req, res) => {
  const { role, skill } = req.query;
  if (skill && role) {
    let [result] = await db.query(
      "SELECT * FROM users WHERE role = ? AND JSON_CONTAINS(skills, ?)",
      [role, skill]
    );
    result = await decodeAndParseFields(result);
    for (const user of result) {
      const id = user.id;
      const [total] = await db.query(
        `SELECT COUNT(*) AS total FROM custom_task_creation WHERE responsible_person = '${id}' AND deleted = '0'`
      );
      user.task = total[0]?.total ? total[0]?.total : 0;
    }
    return sendResponse(res, 200, result);
  } else {
    return sendResponse(res, 200, []);
  }
};

export const kanbanStatusUpdate = async (req, res) => {
  const { id, status } = req.body;
  await db.query(
    `UPDATE custom_task_creation SET status = '${status}' WHERE id = '${id}'`
  );

  return sendResponse(res, 200, "Record updated successfully");
};

export const getKanbanData = async (req, res) => {
  req.query.filter = JSON.parse(req.query.filter);
  let { organization, date } = req.query.filter;

  const data = [
    {
      id: "1",
      status: "To-Do",
      title: "🏢 To Do",
      tasks: [],
    },
    {
      id: "2",
      status: "In-Progress",
      title: "👨‍🏭 In Progress",
      tasks: [],
    },
    {
      id: "3",
      status: "Completed",
      title: "✅ Completed",
      tasks: [],
    },
  ];

  let condition = "";
  for (const ele of Object.entries(req.query.filter)) {
    if (ele[1] != "" && ele[1] != null) {
      ele[1] = Number(ele[1]);
      condition += ` AND ${ele[0]} = '${ele[1]}'`;
    }
  }
  // console.log(condition);
  let [result] = await db.query(
    `SELECT * FROM custom_task_creation WHERE deleted = 0 ${condition}`
  );
  // console.log(req.query);
  // console.log(organization);
  // console.log(result);
  result = await decodeAndParseFields(result);

  for (const element of result) {
    let arr = [];
    const collaborators = element.collaborators;
    if (collaborators) {
      // console.log(collaborators);
      for (const collaborator of collaborators) {
        const [user] = await db.query(
          `SELECT id , CONCAT(name , ' ' , surname) AS name , profile FROM users WHERE id = '${collaborator}' AND deleted = '0'`
        );
        // console.log(user);
        arr.push({
          id: user[0]?.id,
          name: user[0]?.name,
          profile: user[0]?.profile,
        });
      }
    }
    // console.log(arr);
    const index = data.findIndex((d) => d.status === element.status);
    data[index]?.tasks.push(element);
    element.collaborator_details = arr;
  }
  return sendResponse(res, 200, data);
};
