import asyncHandler from "express-async-handler";
import db from "../db-config.js";
import {
  insertActivityLog,
  storeError,
  getOrganizationAccordingToDepartment,
  whereCondition,
  makeJoins,
  countQueryCondition,
  deleteRecord,
  employeeTree,
  decodeSingle_statement,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  getRecord,
  decodeAndParseFields,
  processesSingleDDRMDocument,
  uniqueIdGenerator,
  getUserById,
  sendDynamicEmail,
} from "../helper/general.js";
import HSEAppointment from "../sequelize/HSEAppointmentSchema.js";
import { sendResponse } from "../helper/wrapper.js";
import { uploadToDDRM } from "../helper/ddrmUploader.js";
import sendEmail from "../helper/sendEmail.js";

/**function to create and appointed_user_id HseAppointment */
export const createUpdateHseAppointment = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];

  const {
    id,
    appoints_user_id,
    legal_appointment_role,
    appointed_user_id,
    name_of_appointment,
    effective_date,
    end_date,
    organization,
    department,
    sidebar_id = 96,
  } = req.body;

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  // already exist between effective date throw error;
  const idCheck = id ? `AND hse_appointment.id != ${id}` : "";
  const [appointmentRecord] = await getRecord(
    "appointment_type",
    "id",
    name_of_appointment
  );
  // return
  const hseQuery = `
  SELECT * 
  FROM hse_appointment 
  LEFT JOIN appointment_type ON hse_appointment.name_of_appointment = appointment_type.id 
  WHERE hse_appointment.deleted = 0 
  AND hse_appointment.end_date >= ? 
  AND hse_appointment.organization = ?
  AND appointment_type.name = '${appointmentRecord?.name.trim()}'
  ${idCheck}
  `;
  const [record] = await db.query(hseQuery, [
    effective_date,
    req.body.organization,
  ]);
  if (record.length > 0 && appointmentRecord?.name?.trim() == "16 (1)") {
    return sendResponse(
      res,
      400,
      "16 (1) appointment already exist between the selected date"
    );
  }

  if (req.files?.appointer_signature) {
    const file = req.files.appointer_signature;
    const ddrm_id = await processesSingleDDRMDocument(
      "hse_appointment",
      sidebar_id,
      file,
      req
    );
    req.body.appointer_ddrm_id = ddrm_id;
  }
  if (req.files?.appointed_signature) {
    const file = req.files.appointed_signature;
    const ddrm_id = await processesSingleDDRMDocument(
      "hse_appointment",
      sidebar_id,
      file,
      req
    );
    req.body.appointed_ddrm_id = ddrm_id;
  }

  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    /**Update HseAppointment Query */
    const { query, values } = updateQueryBuilder(HSEAppointment, req.body);
    await db.query(query, values);

    /**Insert Activity  */
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "hse_appointment",
      id
    );

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "HSE",
      "hse_appointment",
      "unique_id",
      "unique_id"
    );
    req.body.unique_id = unique_id;
    req.body.created_by = req.user.sessionid;
    /**Insert record for HSE Appointment */
    const { query, values } = createQueryBuilder(HSEAppointment, req.body);
    const createHseAppointment = await db.query(query, values);

    // send email to appointer and appointed user
    const appointed = await getUserById(appointed_user_id);
    const appointer = await getUserById(appoints_user_id);

    const companyWebsite = process.env.HARMONY_FRONTEND_URL;

    const sendRecordArray = {
      templateFileUrl: "mail_for_hse_appointer_template.html",
      appointer_name: appointer.name,
      appointed_employee_name: appointed.name,
      templateName: "HSE Appointer",
      company_name: appointer.company_name,
      company_website: companyWebsite,
      HSE_Legal_Appointment_Name: `${appointmentRecord?.name.trim()}`,
      effective_date: effective_date,
      end_date: end_date,
    };

    // send to the appointer
    await sendDynamicEmail({
      to: appointer.email,
      subject: `Confirmation: HSE Appointment - ${appointed.name}`,
      data: sendRecordArray,
    });

    sendRecordArray.templateFileUrl = "mail_for_hse_appointed_template.html";
    sendRecordArray.templateName = "HSE Appointed";

    // send to the appointed
    await sendDynamicEmail({
      to: appointed.email,
      subject: `HSE Appointment Notification - ${appointmentRecord?.name.trim()}`,
      data: sendRecordArray,
    });

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "hse_appointment",
      createHseAppointment.insertId
    );

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**function to get all HseAppointment or specific HseAppointment */
export const getHseAppointment = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "hse_appointment",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "CONCAT(u1.name , ' ' , u1.surname)",
    "CONCAT(u2.name , ' ' , u2.surname)",
    "hse_appointment.effective_date",
    "r1.name",
    "r2.name",
    "organization.name",
    "users.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = hse_appointment.created_by",
    },
    {
      type: "left",
      targetTable: "users u1",
      onCondition: "u1.id = hse_appointment.appoints_user_id",
    },
    {
      type: "left",
      targetTable: "users u2",
      onCondition: "u2.id = hse_appointment.appointed_user_id",
    },
    {
      type: "left",
      targetTable: "roles r1",
      onCondition: " r1.id = u1.role",
    },
    {
      type: "left",
      targetTable: "roles r2",
      onCondition: " r2.id = u2.role",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: " organization.id = hse_appointment.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: " department.id = hse_appointment.department",
    },
    {
      type: "left",
      targetTable: "department as d1",
      onCondition: " d1.id = u1.department",
    },
    {
      type: "left",
      targetTable: "department as d2",
      onCondition: " d2.id = u2.department",
    },
    {
      type: "left",
      targetTable: "appointment_type",
      onCondition: " appointment_type.id = hse_appointment.name_of_appointment",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = hse_appointment.physical_location",
    },
    {
      type: "left",
      targetTable: "repository",
      onCondition: " repository.id = hse_appointment.ddrm_id",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all hse appointment */
  const hseAppointmentQuery = `SELECT appointment_type.assignment,hse_appointment.unique_id ,hse_appointment.signature_type, appointment_type.designated, appointment_type.applicable, repository.url as signature, u1.profile as appoints_profile , u2.profile as appointed_profile,
  location.name as physical_location_name , hse_appointment.organization , hse_appointment.department, organization.name as organization_name, organization.header_image , organization.footer_image , organization.business_logo, department.name as department_name , hse_appointment.id , hse_appointment.effective_date , hse_appointment.end_date , hse_appointment.physical_location, hse_appointment.created_by as created_by_id , users.name as created_by , users.surname as created_by_surname , users.profile as created_by_profile , u1.id as appoints_id , u1.unique_id as appoints_unique_id , u1.name as appoints_name ,u1.surname as appoints_sur_name , u2.id as appointed_id , u2.unique_id as appointed_unique_id , u2.name as appointed_name ,u2.surname as appointed_sur_name ,  d1.name as appointer_department_name, d2.name as appointed_department_name, 
    hse_appointment.name_of_appointment as name_of_appointment_id ,
    appointment_type.name as name_of_appointment,
    r1.name as appoints_role ,
    r2.name as appointed_role
        FROM hse_appointment 
        ${joinsRecord} where hse_appointment.deleted = 0  ${searchCondition} ${condition}`;
  const [hseAppointment] = await db.query(hseAppointmentQuery);
  // hse_role.name as appoints_legal_role_name ,
  // hse_role.id as appoints_legal_role_id,
  // l2.name as appointed_legal_role_name ,
  // l2.id as appointed_legal_role_id ,
  for (let i = 0; i < hseAppointment.length; i++) {
    hseAppointment[i].assignment = await decodeSingle_statement(
      hseAppointment[i].assignment
    );

    hseAppointment[i].designated = await decodeSingle_statement(
      hseAppointment[i].designated
    );

    hseAppointment[i].applicable = await decodeSingle_statement(
      hseAppointment[i].applicable
    );
  }

  const totalRecord = await countQueryCondition(hseAppointmentQuery);

  return sendResponse(res, 200, hseAppointment, totalRecord);
};

/**Function to delete a specific HseAppointment */
export const deleteHseAppointment = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("hse_appointment", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Hse Appointment",
      id
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
};

export const getHseAppointmentChartOld = async (req, res) => {
  /** Get top level person of an organization  using query*/
  const where = req.query.organization
    ? `AND u.my_organization = ${req.query.organization}`
    : `AND u.my_organization = ${req.user.sessionMyOrganization}`;
  const levelOnePersonQuery = `SELECT u.id, u.name, u.profile,  roles.name as position,role_hierarchy.level AS hierarchy_level FROM users u LEFT JOIN roles on roles.id = u.role LEFT JOIN role_hierarchy on role_hierarchy.id = roles.hierarchy WHERE u.deleted = "0" ${where} ORDER BY role_hierarchy.level ASC, u.id ASC;`;
  const [levelOnePersonList] = await db.query(levelOnePersonQuery);
  // return
  // let chartStructure = {
  //   id: 0,
  //   name: "Harmony & Help",
  //   position: "Harmony & Help",
  //   profile: "harmony.png",
  //   children: [],
  // };
  /** if organization filter is added then Change the chartStructure Object name as per organization */
  let chartStructure;
  if (req.query.organization || req.user.sessionMyOrganization) {
    const [organization] = await getRecord(
      "organization",
      "id",
      req.query.organization || req.user.sessionMyOrganization
    );
    //
    chartStructure = {
      id: organization?.id,
      name: organization?.name,
      position: "Harmony & Help",
      profile: "harmony.png",
      children: [],
    };

    // Group users by their hierarchy levels
    const groupedUsers = levelOnePersonList.reduce((acc, user) => {
      const level = user.hierarchy_level;
      if (!acc[level]) acc[level] = [];
      acc[level].push({
        id: user.id,
        name: user.name,
        position: user.position,
        profile: user.profile,
        // children: [],
      });
      return acc;
    }, {});

    chartStructure.children = Object.keys(groupedUsers)
      .sort((a, b) => a - b)
      .map((level) => ({
        level: parseInt(level),
        users: groupedUsers[level],
      }));
  }
  // let hierarchyObj = {
  //   children: [],
  // };
  // for (let user of levelOnePersonList) {
  //   hierarchyObj = {
  //     ...user,
  //     children: [],
  //   };
  //   const allTeamQuery = `SELECT t.id, t.name, t.team_leader, t.team_members, t.organization,
  //             t.department,u.name as team_leader_name,u.profile, u.role, roles.name as role_name FROM users u
  //            LEFT JOIN teams t ON u.id = t.team_leader
  //            LEFT JOIN roles on roles.id = u.role
  //            WHERE NOT EXISTS ( SELECT 1 FROM teams tm WHERE JSON_CONTAINS(tm.team_members, CAST(u.id AS CHAR), '$'))`;
  //   let [allTeamList] = await db.query(allTeamQuery);
  //   /** null check for allTeamList */
  //   allTeamList = allTeamList.filter((e) => e.id);
  //   for (let i = 0; i < allTeamList.length; i++) {
  //     const team = allTeamList[i];
  //     const child = await employeeTree(team);
  //     hierarchyObj.children.push(child);
  //   }
  // }
  // if (hierarchyObj.id) {
  //   chartStructure.children.push(hierarchyObj);
  // }
  return sendResponse(res, 200, chartStructure);
};

export const getHseAppointmentChart = async (req, res) => {
  /** Get top level person of an organization  using query*/
  const where = req.query.organization
    ? `AND u.my_organization = ${req.query.organization}`
    : `AND u.my_organization = ${req.user.sessionMyOrganization}`;
  const levelOnePersonQuery = `SELECT u.id, u.name, u.profile,  roles.name as position,role_hierarchy.level AS hierarchy_level FROM users u LEFT JOIN roles on roles.id = u.role LEFT JOIN role_hierarchy on role_hierarchy.id = roles.hierarchy WHERE u.deleted = "0" AND role_hierarchy.level = 1  ${where} ORDER BY role_hierarchy.level ASC, u.id ASC;`;
  const [levelOnePersonList] = await db.query(levelOnePersonQuery);
  // return
  // let chartStructure = {
  //   id: 0,
  //   name: "Harmony & Help",
  //   position: "Harmony & Help",
  //   profile: "harmony.png",
  //   children: [],
  // };
  /** if organization filter is added then Change the chartStructure Object name as per organization */
  let chartStructure;
  if (req.query.organization || req.user.sessionMyOrganization) {
    const [organization] = await getRecord(
      "organization",
      "id",
      req.query.organization || req.user.sessionMyOrganization
    );
    //
    chartStructure = {
      id: organization?.id,
      name: organization?.name,
      role_name: "Organization",
      profile: organization?.business_logo || "harmony.png",
      department: "Harmony & Help",
      members: [],
    };

    /// all employees other than top level
    const otherLevelQuery = `SELECT u.id, u.name, u.profile, u.manager, department.name as department,  roles.name as role_name ,role_hierarchy.level FROM users u LEFT JOIN roles on roles.id = u.role 
    LEFT JOIN department on department.id = u.department
    LEFT JOIN role_hierarchy on role_hierarchy.id = roles.hierarchy WHERE u.deleted = "0"  ${where} ORDER BY role_hierarchy.level ASC, u.id ASC;`;
    const [otherLevelPersonList] = await db.query(otherLevelQuery);

    function buildHierarchy(allEmployees, levelOnePersonList) {
      if (!Array.isArray(allEmployees) || !Array.isArray(levelOnePersonList))
        return null;

      const hierarchyMap = new Map();

      // Initialize all employees in the map
      allEmployees.forEach((person) => {
        hierarchyMap.set(person.id, { ...person, members: [] });
      });

      let rootEmployees = [];

      // Build the tree by linking employees under their managers
      allEmployees.forEach((person) => {
        if (person.manager && hierarchyMap.has(person.manager)) {
          hierarchyMap
            .get(person.manager)
            .members.push(hierarchyMap.get(person.id));
        } else {
          if (hierarchyMap.get(person.id).level == 1) {
            rootEmployees.push(hierarchyMap.get(person.id)); // If no manager found, it's a root employee
          }
        }
      });

      return rootEmployees;
    }

    const hierarchy = buildHierarchy(otherLevelPersonList, levelOnePersonList);
    if (hierarchy) {
      chartStructure.members.push(...hierarchy);
    }
  }

  return sendResponse(res, 200, chartStructure);
};

export const uploadSignature = async (req, res) => {
  const { sidebar_id, id, ddrm_id } = req.body;
  //
  // const signature = req.files.signature;
  // const ddrm_id = req.files.ddrm_id;
  if (!ddrm_id) {
    return sendResponse(res, 400, "DDRM id is required");
  }

  // const ddrm_id = await uploadToDDRM(sidebar_id, signature, req);

  const { query, values } = updateQueryBuilder(HSEAppointment, {
    id,
    ddrm_id,
  });

  const [result] = await db.query(query, values);
  if (result.affectedRows > 0) {
    return sendResponse(res, 200, "Signature uploaded successfully");
  } else {
    return sendResponse(res, 400, "Error uploading signature");
  }
};
