import {
  countQueryCondition,
  decodeAndParseFields,
  decodeSingle_statement,
  deleteRecord,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  processUploadDocuments,
  searchConditionRecord,
  uniqueIdGenerator,
  uploadFile,
  whereCondition,
} from "../helper/general.js";
import db from "../db-config.js";
import {
  createQueryBuilder,
  updateQueryBuilder,
} from "../helper/queryBuilder.js";
import { sendResponse } from "../helper/wrapper.js";
import InsuranceClaim from "../sequelize/InsuranceClaimRecordingSchema.js";
import sendEmail from "../helper/sendEmail.js";
import InsuranceClaimDocs from "../sequelize/InsuranceClaimDocsSchema.js";

/**Function to create Safety Incident Form  */
export const createUpdateInsuranceClaim = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];
  const {
    id,
    department,
    description,
    comments,
    sidebar_id = 191,
    upload_documents,
  } = req.body;
  const status = id ? "updated" : "created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  req.body.description = await encodeSingle_statement(description);
  let organizationId = req.body.organization;

  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0]?.organization;
    if (organizationId) {
      req.body.organization = organizationId;
    }
  }

  if (!id) {
    req.body.unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "IC",
      "insurance_claim",
      "unique_id"
    );
    req.body.reference_no = await uniqueIdGenerator(
      req.body.organization,
      department,
      "IC",
      "insurance_claim",
      "reference_no",
      "reference"
    );
  }

  const documents =
    upload_documents || (await processUploadDocuments(req, sidebar_id));

  if (id && documents.length) {
    const deleteQuery = `UPDATE insurance_claim_docs SET deleted = 1 where deleted = 0 AND insurance_claim_id = ?`;
    await db.query(deleteQuery, [id]);
  }

  req.body.internal_insurance_claim_id = await uniqueIdGenerator(
    req.body.organization,
    department,
    "IC",
    "insurance_claim",
    "internal_insurance_claim_id"
  );
  req.body.comments = await encodeSingle_statement(comments);
  const { query, values } = id
    ? updateQueryBuilder(InsuranceClaim, req.body)
    : createQueryBuilder(InsuranceClaim, req.body);
  const [result] = await db.query(query, values);

  if (documents || Array.isArray(documents)) {
    for (const file of documents) {
      const title = file.title; // Use file name as title
      const ddrm_id = file?.ddrm_id;
      const id_doc = file.id;
      // Move the file to the public folder
      // Save file details for database
      const { query: docQuery, values: docValues } = createQueryBuilder(
        InsuranceClaimDocs,
        {
          doc_no: id_doc,
          insurance_claim_id: id ? id : result.insertId,
          title,
          ddrm_id,
        }
      );

      await db.query(docQuery, docValues);
    }
  }

  await insertActivityLog(
    req.user.sessionid,
    "create",
    "Insurance claim",
    `This user ${status} a new Insurance Claim  Record for organization ${organizationId}`
  );

  return sendResponse(res, 200, `Record ${status} successfully`);
};

export const getInsuranceClaim = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "insurance_claim",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
    columnName: req.query.type,
  });

  const searchTableName = [
    "insurance_claim.incident_report_id",
    "insurance_claim.description",
    "insurance_claim.incident_category",
    "insurance_claim.initial_incident_severity",
    "insurance_claim.location",
    "insurance_claim.insurance_policy_impact",
    "insurance_claim.currency",
    "insurance_claim.claim_number",
    "incident.incident_title",
    "insurance_claim.insurance_claim_approval_status",
    "insurance_claim.comments",
    "organization.name",
    "department.name",
    "CONCAT(users.name , ' ' , users.surname)",
  ];

  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = insurance_claim.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = insurance_claim.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = insurance_claim.department",
    },
    {
      type: "left",
      targetTable: "incident_category",
      onCondition: "incident_category.id = insurance_claim.incident_category",
    },
    {
      type: "left",
      targetTable: "incident",
      onCondition: "incident.id = insurance_claim.incident_report_id",
    },
    {
      type: "left",
      targetTable: "users AS responsible",
      onCondition: "responsible.id = insurance_claim.responsible_person",
    },
    {
      type: "left",
      targetTable: "roles",
      onCondition: "roles.id  = responsible.role",
    },
  ];

  // description
  // incident_category
  // date_of_incident
  // time_of_incident_occurrence
  // initial_incident_severity
  // location

  const joinCondition = await makeJoins(joins);
  const insuranceClaimFormQuery = `SELECT insurance_claim.* , organization.name as organization_name ,department.name as department_name , users.name as created_by_name , users.surname as created_by_surname  , users.profile as created_by_profile , incident.incident_title as incident_report_name , incident.incident_location AS location , incident.description AS description , incident.incident_category AS incident_category , incident.incident_reporting_date AS date_of_incident  , incident.incident_actual_time AS time_of_incident_occurrence , incident.initial_incident_severity AS initial_incident_severity , CONCAT(responsible.name , ' ' , responsible.surname) AS responsible_person_name , roles.name AS responsible_person_role  FROM insurance_claim ${joinCondition} WHERE insurance_claim.deleted = 0 ${searchCondition}  ${condition} `;
  const totalRecord = await countQueryCondition(insuranceClaimFormQuery);

  let [insuranceClaimForm] = await db.query(insuranceClaimFormQuery);
  insuranceClaimForm = await decodeAndParseFields(insuranceClaimForm);
  for (const data of insuranceClaimForm) {
    // add incident location name
    const incidentLocationIds = data.location;
    let record = [];
    if (incidentLocationIds && incidentLocationIds.length) {
      [record] = await db.query(
        `SELECT name FROM location WHERE id IN (${incidentLocationIds})`
      );
    }
    const locationName = record.map((item) => item.name);
    data.location_name = locationName;
    const [files] = await db.query(
      `SELECT insurance_claim_docs.*, repository.url as file, document_creation.document_name as title FROM insurance_claim_docs LEFT JOIN repository ON repository.id = insurance_claim_docs.ddrm_id LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE insurance_claim_docs.deleted = 0 AND insurance_claim_docs.insurance_claim_id = ${data.id}`
    );
    console.log(
      `SELECT insurance_claim_docs.*, repository.url as file, document_creation.document_name as title FROM insurance_claim_docs LEFT JOIN repository ON repository.id = insurance_claim_docs.ddrm_id LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE insurance_claim_docs.deleted = 0 AND insurance_claim_docs.insurance_claim_id = ${data.id}`
    );

    data.upload_documents = files;

    const arr = [];
    if (data.incident_category && Array.isArray(data.incident_category)) {
      for (const i of data.incident_category) {
        const obj = {};
        const [fetchIncident] = await db.query(
          `SELECT id , name FROM incident_category WHERE id = ${i}`
        );
        obj.id = fetchIncident[0]?.id;
        obj.name = fetchIncident[0]?.name;
        arr.push(obj);
      }
      data.incident_category_details = arr;
    }
    const arr2 = [];
    if (
      data.insurance_policy_impact &&
      Array.isArray(data.insurance_policy_impact)
    ) {
      for (const i of data.insurance_policy_impact) {
        const obj = {};
        const [fetchInsurance] = await db.query(
          `SELECT id , name FROM insurance_policy_impact WHERE id = ${i}`
        );
        obj.id = fetchInsurance[0]?.id;
        obj.name = fetchInsurance[0]?.name;
        arr2.push(obj);
      }
      data.insurance_policy_impact_details = arr2;
    }
  }

  return sendResponse(res, 200, insuranceClaimForm, totalRecord);
};

export const deleteInsuranceClaim = async (req, res) => {
  const { id } = req.params;
  const deleteForm = await deleteRecord("insurance_claim", id);
  if (deleteForm) {
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Insurance Claim Form",
      `This user deleted a new Insurance Claim Record for organization ${id}`
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
  return sendResponse(res, 404, "Cannot delete record");
};

export const sendInsuranceClaim = async (req, res) => {
  const { id, mails, url, message } = req.body;
  for (const mail of mails) {
    let name = mail.split("@")[0];
    // First letter of the name capital
    name = name
      .split(" ")
      .map((word) => word.charAt(0).toUpperCase() + word.slice(1))
      .join(" ");

    const link = url;

    const sendRecordArray = {
      templateFileUrl: "mail_for_insurance_claim_template.html",
      link: link,
      name: name,
      message: message ? message : "Insurance Claims",
      support_mail: process.env.COMPANY_SUPPORT_MAIL || "info@harmonyandhelp.com",
      templateName: "Insurance Claim",
    };
    const info = await sendEmail(
      "info@harmonyandhelp.com",
      mail,
      "Insurance Claim",
      sendRecordArray
    );
  }
  return sendResponse(res, 200, "Insurance sent to email successfully");
};
