import {
  countQueryCondition,
  createUpdateAction,
  decodeAndParseFields,
  decodeSingle_statement,
  deleteFile,
  deleteRecord,
  encodeAndStringifyFields,
  getEmployeeByIds,
  getIncidentInvestigationData,
  getInvestigationDataIntoCompleted,
  getOrganizationAccordingToDepartment,
  getQualityFormData,
  getRecord,
  incidentCategoryConstant,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  whereCondition,
} from "../helper/general.js";
import Investigation from "../sequelize/InvestigationSchema.js";
import db from "../db-config.js";
import { sendResponse } from "../helper/wrapper.js";
import Incident from "../sequelize/IncidentSchema.js";
import {
  createQueryBuilder,
  updateQueryBuilder,
} from "../helper/queryBuilder.js";

/** Function to create and update investigation  */
export const createUpdateInvestigation = async (req, res) => {
  req.body = (await decodeAndParseFields([req.body]))[0];
  let { id, department, incident_id, status } = req.body;

  const [investigationData] = await db.query(
    `select * from investigation where incident_id = ?`,
    [incident_id]
  );
  if (investigationData.length > 0 && !id) {
    return sendResponse(
      res,
      400,
      "Investigation already created with same incident"
    );
  }
  let statusMessage = id ? "update" : "create";
  // console.log(req.body, "req.body");
  if (department) {
    req.body.organization = (
      await getOrganizationAccordingToDepartment(department)
    )[0]?.organization;
  }

  if (!id) {
    req.body.unique_id = await uniqueIdGenerator(
      req.body.organization,
      department,
      "INV",
      "investigation",
      "unique_id",
      "unique_id"
    );
    req.body.reference_no = await uniqueIdGenerator(
      req.body.organization,
      department,
      "INV",
      "investigation",
      "reference_no",
      "reference"
    );
  }

  if (status === "Complete") {
    req.body.is_done = 1;
  } else {
    req.body.is_done = 0;
  }
  const investigation_standardise = req.body?.investigation_standardise || [];
  const investigation_standardise_with_action = await createUpdateAction(
    investigation_standardise,
    req.body.organization,
    req.user.sessionid
  );
  req.body.investigation_standardise = investigation_standardise_with_action;
  req.body = await encodeAndStringifyFields(req.body);
  // handle action replication here

  // let result = id
  //   ? await Investigation.update(req.body, {
  //       where: { id },
  //     })
  //   : await Investigation.create(req.body);

  const { query, values } = id
    ? updateQueryBuilder(Investigation, req.body)
    : createQueryBuilder(Investigation, req.body);
  let [result] = await db.query(query, values);
  // await Incident.update(
  //   {
  //     incident_persons_involved: req.body.incident_persons_involved,
  //   },
  //   {
  //     where: { id: incident_id },
  //   }
  // );

  if (status === "Complete") {
    await Incident.update(
      {
        is_done: 1,
      },
      {
        where: { id: incident_id },
      }
    );
  }
  await insertActivityLog(
    req.user.sessionid,
    statusMessage,
    "Stakeholder Identification",
    id ? id : result.id
  );

  return sendResponse(res, 200, `Record ${status} successfully`);
};

/** Function to get Investigation */
export const getInvestigation = async (req, res) => {
  const { id } = req.params;
  const condition = await whereCondition({
    table: "investigation",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "investigation.incident_name",
    "investigation.initial_incident_severity",
    "investigation.incident_category",
    "investigation.incident_description",
    "investigation.incident_location",
    "investigation.role_in_incident",
    "investigation.investigator_name",
    "investigation.specialty",
    "investigation.organization_external",
    "investigation.person_involved_internal_external",
    // "investigation.involved_person",
    "investigation.involved_role",
    "investigation.involved_department",
    "investigation.involved_role_in_incident",
    "investigation.investigation_enquiry_outcome",
    "investigation.recall_required",
    "investigation.applicable_sop_created",
    "investigation.affected_employees_retrained",
    "investigation.recommended_can_be_applied",
    "investigation.role_in_incident_external",
    "investigation.investigation_organization",
    "investigation.assets_damage_assets_property_type",
    "investigation.assets_currency",
    "investigation.assets_vehicle_incident_type",
    "investigation.quality_incident_type",
    "investigation.customer_impact",
    "investigation.product_impact",
    "investigation.quality_currency",
    "investigation.quality_vehicle_incident_type",
    "investigation.environment_consequence_type",
    "investigation.origin",
    "investigation.injured_person",
    "investigation.injury_classification",
    "investigation.part_body_affected",
    "investigation.reportable_classification",
    "investigation.exposed_person",
    "investigation.diseases_illnesses_type",
    "investigation.exposure_route",
    "investigation.security_incident_type",
    "investigation.security_currency",
    "investigation.line_manager_type",
    "(created_user.name , ' ' , created_user.surname)",
    "organization.name",
  ];

  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      targetTable: "department",
      onCondition: "investigation.department = department.id",
      type: "LEFT",
    },
    {
      targetTable: "organization",
      onCondition: "investigation.organization = organization.id",
      type: "LEFT",
    },
    {
      targetTable: "users as employeeUser",
      onCondition: "investigation.employee = employeeUser.id",
      type: "LEFT",
    },

    {
      targetTable: "users as created_user",
      onCondition: "investigation.created_by = created_user.id",
      type: "LEFT",
    },
    {
      type: "LEFT",
      targetTable: "department AS employee_user_department",
      onCondition: "employeeUser.department = employee_user_department.id",
    },
    {
      type: "LEFT",
      targetTable: "roles AS employee_user_role",
      onCondition: "employeeUser.role = employee_user_role.id",
    },
    {
      type: "left",
      targetTable: "incident",
      onCondition: "incident.id = investigation.incident_id",
    },
    {
      type: "left",
      targetTable: "severity",
      onCondition: "severity.id = investigation.actual_incident_severity",
    },
    // {
    //   type: "left",
    //   targetTable: "users as involved_person",
    //   onCondition: "involved_person.id = investigation.involved_person",
    // },
    {
      type: "left",
      targetTable: "repository as investigation_ddrm",
      onCondition:
        "investigation_ddrm.id = investigation.investigation_ddrm_id",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  const investigationGetQuery = `
    SELECT investigation.*, 
           incident.incident_title AS incident_title_name,
           employeeUser.name AS employee_name, 
           employeeUser.role AS employee_role,
           employeeUser.department AS employee_department,
           severity.name AS actual_incident_severity_name,
           employee_user_department.name AS employee_department_name,
           employee_user_role.name AS employee_role_name,
           created_user.name AS created_by_name, 
           created_user.surname AS created_by_surname, 
           organization.name AS organization_name,
           department.name AS department_name ,
           investigation_ddrm.url AS investigation_ddrm_file
    FROM investigation 
    ${joinsRecord}
    WHERE investigation.deleted = 0 ${searchCondition} ${condition}`;

  let [getInvestigationRecord] = await db.query(investigationGetQuery);
  getInvestigationRecord = await decodeAndParseFields(getInvestigationRecord);
  // console.log("getInvestigationRecord: ", getInvestigationRecord);

  for (let i = 0; i < getInvestigationRecord.length; i++) {
    const investigation = getInvestigationRecord[i];
    if (investigation.incident_persons_involved && Array.isArray(investigation.incident_persons_involved)) {
      for (let i = 0; i < investigation.incident_persons_involved.length; i++) {
        const person = investigation.incident_persons_involved[i];

        if (person.person_type === "Internal") {
          if (person.user_id) {
            const [fetch] = await db.query(
              `SELECT CONCAT(users.name, ' ', users.surname) AS user_name, 
                      roles.name AS role_name, 
                      department.name AS department_name 
               FROM users 
               LEFT JOIN roles ON users.role = roles.id 
               LEFT JOIN department ON users.department = department.id 
               WHERE users.id = ${person.user_id}`
            );

            let incidentRole = [];
            if (person.internal_incident_role) {
              [incidentRole] = await db.query(
                `SELECT name as internal_incident_role_name 
                 FROM incident_role 
                 WHERE id = ${person.internal_incident_role}`
              );
            }

            investigation.incident_persons_involved[i] = {
              ...person,
              ...(fetch[0] || {}),
              ...(incidentRole[0] || {})
            };
          }
        }

        else if (person.person_type === "External") {
          if (person.is_contractor === '1' && person.contractor_id) {
            const [contractorData] = await db.query(
              `SELECT contractor_name as contractor_name 
               FROM contractor_registration 
               WHERE id = ${person.contractor_id}`
            );

            const [incidentRole] = await db.query(
              `SELECT name as external_incident_role_name 
               FROM incident_role 
               WHERE id = ${person.external_incident_role}`
            );

            let projectData = {};
            if (person.project_id) {
              const [project] = await db.query(
                `SELECT id as project_id, project_title as project_name 
                 FROM project_registration 
                 WHERE id = ${person.project_id}`
              );
              projectData = project[0] || {};
            }

            investigation.incident_persons_involved[i] = {
              ...person,
              ...(contractorData[0] || {}),
              ...(incidentRole[0] || {}),
              ...projectData
            };
          } else {
            const [incidentRole] = await db.query(
              `SELECT name as external_incident_role_name 
               FROM incident_role 
               WHERE id = ${person.external_incident_role}`
            );

            investigation.incident_persons_involved[i] = {
              ...person,
              ...(incidentRole[0] || {})
            };
          }
        }

        else if (person.person_type === "Both") {
          let userDetails = {};
          if (person.user_id) {
            const [fetch] = await db.query(
              `SELECT CONCAT(users.name, ' ', users.surname) AS user_name, 
                      roles.name AS role_name, 
                      department.name AS department_name 
               FROM users 
               LEFT JOIN roles ON users.role = roles.id 
               LEFT JOIN department ON users.department = department.id 
               WHERE users.id = ${person.user_id}`
            );
            userDetails = fetch[0] || {};
          }

          let internalRoleDetails = {};
          if (person.internal_incident_role) {
            const [internalRole] = await db.query(
              `SELECT name as internal_incident_role_name 
               FROM incident_role 
               WHERE id = ${person.internal_incident_role}`
            );
            internalRoleDetails = internalRole[0] || {};
          }

          let externalDetails = {};
          let projectDetails = {};

          let externalRoleDetails = {};
          if (person.external_incident_role) {
            const [externalRole] = await db.query(
              `SELECT name as external_incident_role_name 
               FROM incident_role 
               WHERE id = ${person.external_incident_role}`
            );
            externalRoleDetails = externalRole[0] || {};
          }

          if (person.is_contractor === '1' && person.contractor_id) {
            const [contractorData] = await db.query(
              `SELECT contractor_name as contractor_name 
               FROM contractor_registration 
               WHERE id = ${person.contractor_id}`
            );
            externalDetails = contractorData[0] || {};

            if (person.project_id) {
              const [project] = await db.query(
                `SELECT id as project_id, project_title as project_name 
                 FROM project_registration 
                 WHERE id = ${person.project_id}`
              );
              projectDetails = project[0] || {};
            }
          }

          investigation.incident_persons_involved[i] = {
            ...person,
            ...userDetails,
            ...internalRoleDetails,
            ...externalRoleDetails,
            ...externalDetails,
            ...projectDetails
          };
        }
      }
    }

    if (investigation.investigation_enquiry_details && Array.isArray(investigation.investigation_enquiry_details)) {
      for (let i = 0; i < investigation.investigation_enquiry_details.length; i++) {
        const person = investigation.investigation_enquiry_details[i];
        if (person.person_involved_internal_external === "0") {
          investigation.investigation_enquiry_details[i] = {
            ...person,
          };
        } else if (person.person_involved_internal_external === "1") {
          let userDetails = {};
          if (person.involved_person && typeof person.involved_person == "number") {
            const [fetch] = await db.query(
              `SELECT CONCAT(users.name, ' ', users.surname) AS involved_person, 
                      roles.name AS role_name, 
                      department.name AS department_name 
               FROM users 
               LEFT JOIN roles ON users.role = roles.id 
               LEFT JOIN department ON users.department = department.id 
               WHERE users.id = ${person.involved_person}`
            );
            userDetails = fetch[0] || {};
          }
          investigation.investigation_enquiry_details[i] = {
            ...person,
            ...userDetails,
          };
        }
      }
    }
    if (
      investigation?.evidence_collected &&
      investigation?.evidence_collected.length > 0 &&
      investigation?.evidence_collected[0]?.ddrm_id
    ) {
      for (let evidence of investigation?.evidence_collected) {
        const [repository] = await db.query(
          `SELECT repository.url, document_name as title FROM repository LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE repository.id = ?`,
          [evidence.ddrm_id]
        );
        evidence.file = repository[0]?.url;
        evidence.title = repository[0]?.title;
      }
    }
    if (
      investigation?.upload_document &&
      investigation?.upload_document.length > 0 &&
      investigation?.upload_document[0]?.ddrm_id
    ) {
      for (let uploadDoc of investigation?.upload_document) {
        const [repository] = await db.query(
          `SELECT repository.url, document_name as title FROM repository LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE repository.id = ?`,
          [uploadDoc.ddrm_id]
        );
        uploadDoc.file = repository[0]?.url;
        uploadDoc.title = repository[0]?.title;
      }
    }

    if (investigation?.type_of_security_incident) {
      const [securityIncidentType] = await db.query(
        "SELECT name FROM security_incident_type WHERE id = ?",
        [investigation.type_of_security_incident]
      );
      investigation.type_of_security_incident_name =
        securityIncidentType[0]?.name;
    }

    if (
      investigation &&
      investigation?.any_arrests &&
      investigation?.any_arrests.length > 0
    ) {
      const [users] = await db.query(
        `SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
          LEFT JOIN department ON department.id  = users.department WHERE users.id IN (?)`,
        [investigation.any_arrests]
      );
      investigation.any_arrests_details = users;
    }

    if (investigation?.security_currency) {
      const [securityCurrency] = await db.query(
        "SELECT name FROM currency WHERE id = ?",
        [investigation.security_currency]
      );
      investigation.security_currency_name = securityCurrency[0]?.name;
    }
    if (investigation?.estimated_currency) {
      const [securityCurrency] = await db.query(
        "SELECT name FROM currency WHERE id = ?",
        [investigation.estimated_currency]
      );
      investigation.estimated_currency_name = securityCurrency[0]?.name;
    }
    // for incident category name
    if (
      investigation?.incident_category &&
      investigation?.incident_category.length > 0
    ) {
      const [incidentCategory] = await db.query(
        "SELECT name FROM incident_category WHERE id IN (?)",
        [investigation.incident_category]
      );
      getInvestigationRecord[i].incident_category_name = incidentCategory;
    }

    /// for similar incidents
    if (
      investigation?.similar_incidents &&
      investigation?.similar_incidents.length > 0
    ) {
      const [similarIncidents] = await db.query(
        "SELECT id, incident_title as name FROM incident WHERE id IN (?)",
        [investigation.similar_incidents]
      );
      getInvestigationRecord[i].similar_incidents_name = similarIncidents;

      // for investigation submitted
      if (
        investigation?.investigation_submitted &&
        investigation?.investigation_submitted.length > 0
      ) {
        const investigation_submitted =
          investigation?.investigation_submitted.map((item) => item.employee);
        const investigationSubmitted = await getEmployeeByIds(
          investigation_submitted
        );
        getInvestigationRecord[i].investigation_submitted =
          investigationSubmitted;
      }
    }

    // for notify users
    if (investigation?.notify && investigation?.notify.length > 0) {
      const notify = investigation?.notify.map((item) => item.employee);
      const notifyUsers = await getEmployeeByIds(notify);
      getInvestigationRecord[i].notify = notifyUsers;
    }
    /// actual_incident_category get all related data

    if (
      investigation?.actual_incident_category &&
      investigation?.actual_incident_category.length > 0
    ) {
      const [actualIncidentCategory] = await db.query(
        "SELECT name FROM incident_category WHERE id IN (?)",
        [investigation.actual_incident_category]
      );
      getInvestigationRecord[i].actual_incident_category_name =
        actualIncidentCategory;
    }

    if (investigation.incident_id) {
      const { data } = await getIncidentInvestigationData(
        investigation.incident_id
      );
      getInvestigationRecord[i].incident_data = data?.records;
    }
    // injured_persons , vehicle_details, asset_type , quality_incident_details , incident_type , occupationals
    getInvestigationRecord[i].injured_persons = getInvestigationRecord[i]
      ?.injured_persons
      ? getInvestigationRecord[i].injured_persons
      : [];

    for (let injuredPerson of investigation.injured_persons) {
      if (injuredPerson && injuredPerson?.user_id) {
        const [user] = await db.query(
          `SELECT  CONCAT(name , ' ' ,surname) as name ,profile FROM users WHERE id = ?`,
          [injuredPerson.user_id]
        );
        injuredPerson.user_name = user[0].name;
      }
      if (injuredPerson && injuredPerson?.reported_by) {
        const [user] = await db.query(
          `SELECT  CONCAT(name , ' ' ,surname) as name ,profile FROM users WHERE id = ?`,
          [injuredPerson.reported_by]
        );
        injuredPerson.reported_by_name = user[0].name;
      }

      if (
        injuredPerson &&
        injuredPerson?.parts_of_body_affected &&
        injuredPerson?.parts_of_body_affected.length
      ) {
        const [bodyParts] = await db.query(
          `SELECT label, parent_id FROM body_part WHERE id IN (?)`,
          [injuredPerson.parts_of_body_affected]
        );
        injuredPerson.parts_of_body_affected_details = bodyParts;
      }
    }

    getInvestigationRecord[i].vehicle_details = getInvestigationRecord[i]
      .vehicle_details
      ? getInvestigationRecord[i].vehicle_details
      : [];

    for (let vehicleDetail of investigation.vehicle_details) {
      if (vehicleDetail && vehicleDetail?.reported_employee) {
        const [user] = await db.query(
          `SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id = ?`,
          [vehicleDetail.reported_employee]
        );
        vehicleDetail.reported_employee_name = user[0].name;
        vehicleDetail.reported_employee_profile = user[0].profile;
        vehicleDetail.reported_employee_role = user[0].role;
        vehicleDetail.reported_employee_department = user[0].department;
      }
      if (vehicleDetail && vehicleDetail?.employee) {
        const [user] = await db.query(
          `SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id = ?`,
          [vehicleDetail.employee]
        );
        vehicleDetail.employee_name = user[0].name;
        vehicleDetail.employee_profile = user[0].profile;
        vehicleDetail.employee_role = user[0].role;
        vehicleDetail.employee_department = user[0].department;
      }

      if (vehicleDetail && vehicleDetail?.currency) {
        const [currency] = await db.query(
          `SELECT name FROM currency WHERE id = ?`,
          [vehicleDetail.currency]
        );
        vehicleDetail.currency_name = currency[0]?.name;
      }
      if (vehicleDetail && vehicleDetail?.actual_currency) {
        const [actualCurrency] = await db.query(
          `SELECT name FROM currency WHERE id = ?`,
          [vehicleDetail.actual_currency]
        );
        vehicleDetail.actual_currency_name = actualCurrency[0]?.name;
      }
    }

    getInvestigationRecord[i].asset_type = getInvestigationRecord[i].asset_type
      ? getInvestigationRecord[i].asset_type
      : [];

    for (let assetType of investigation.asset_type) {
      if (assetType && assetType?.select_asset_type) {
        const [asset] = await db.query(
          `SELECT name FROM asset_type WHERE id = ?`,
          [assetType.select_asset_type]
        );
        assetType.select_asset_type_name = asset[0].name;
      }
      if (assetType && assetType?.select_asset) {
        const [asset] = await db.query(
          `SELECT asset_name FROM asset WHERE id = ?`,
          [assetType.select_asset]
        );
        assetType.select_asset_name = asset[0].asset_name;
      }

      if (assetType && assetType?.currency) {
        const [currency] = await db.query(
          `SELECT name FROM currency WHERE id = ?`,
          [assetType.currency]
        );
        assetType.currency_name = currency[0]?.name;
      }
      if (assetType && assetType?.actual_currency) {
        const [actualCurrency] = await db.query(
          `SELECT name FROM currency WHERE id = ?`,
          [assetType.actual_currency]
        );
        assetType.actual_currency_name = actualCurrency[0]?.name;
      }
      if (assetType && assetType?.damage_severity_level) {
        const [severity] = await db.query(
          `SELECT name FROM severity WHERE id = ?`,
          [assetType.damage_severity_level]
        );
        assetType.damage_severity_level_name = severity[0]?.name;
      }

      if (assetType && assetType?.reported_by_whom) {
        const [user] = await db.query(
          `SELECT users.id as user_id ,  CONCAT(users.name , ' ' , users.surname) as name , users.profile, department.name as department , roles.name as role FROM users  LEFT JOIN roles ON roles.id = users.role 
            LEFT JOIN department ON department.id  = users.department WHERE users.id = ?`,
          [assetType.reported_by_whom]
        );
        assetType.reported_by_whom_name = user[0].name;
        assetType.reported_by_whom_profile = user[0].profile;
        assetType.reported_by_whom_role = user[0].role;
        assetType.reported_by_whom_department = user[0].department;
      }
    }

    getInvestigationRecord[i].quality_incident_details = getInvestigationRecord[
      i
    ].quality_incident_details
      ? await getInvestigationRecord[i].quality_incident_details
      : [];

    getInvestigationRecord[i].incident_type = getInvestigationRecord[i]
      .incident_type
      ? getInvestigationRecord[i].incident_type
      : [];

    getInvestigationRecord[i].occupationals = getInvestigationRecord[i]
      .occupationals
      ? getInvestigationRecord[i].occupationals
      : [];

    for (let occupational of investigation.occupationals) {
      if (occupational && occupational?.exposed_person_name) {
        const [user] = await db.query(
          `SELECT id, CONCAT(name , ' ' ,surname) as name ,profile FROM users WHERE id = ?`,
          [occupational.exposed_person_name]
        );
        occupational.exposed_person_id = user[0]?.id;
        occupational.exposed_person_name = user[0]?.name;
        occupational.exposed_person_profile = user[0]?.profile;
      }

      if (
        occupational &&
        occupational?.occupational_type &&
        occupational?.occupational_type.length
      ) {
        const [occupationalType] = await db.query(
          `SELECT name FROM illness_type WHERE id IN (?)`,
          [occupational.occupational_type]
        );
        occupational.occupational_type_details = occupationalType;
      }
      if (occupational && occupational.reported_by) {
        const [user] = await db.query(
          `SELECT CONCAT(name , ' ' ,surname) as name ,profile FROM users WHERE id = ?`,
          [occupational.reported_by]
        );
        occupational.reported_by_name = user[0]?.name;
        occupational.reported_by_profile = user[0]?.profile;
      }
    }

    if (investigation?.investigation_standardise) {
      for (let standardise of investigation?.investigation_standardise) {
        if (standardise && standardise?.who) {
          const [user] = await db.query(
            `SELECT  CONCAT(name , ' ' ,surname) as name ,profile FROM users WHERE id = ?`,
            [standardise.who]
          );
          standardise.who_name = user[0].name;
          standardise.who_profile = user[0].profile;
        }
      }
    }
  }

  const totalCount = await countQueryCondition(investigationGetQuery);

  // console.log(getInvestigationRecord);
  return sendResponse(res, 200, getInvestigationRecord, totalCount);
};

/**Function to delete a specific Investigation */
export const deleteInvestigation = async (req, res) => {
  const { id } = req.params;
  const [filePaths] = await db.query(
    "SELECT * FROM investigation_docs WHERE investigation_id = ?",
    [id]
  );

  if (filePaths?.length > 0) {
    for (const file of filePaths) {
      if (file?.file !== undefined) {
        await deleteFile(file?.file);
      }
    }
  }
  await db.query(
    "UPDATE investigation_docs SET deleted = 1 WHERE investigation_id = ?",
    [id]
  );
  const deleteRecordQuery = await deleteRecord("investigation", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "delete", "Investigation", id);
    return sendResponse(
      res,
      200,
      "Investigation Report Record deleted Successfully"
    );
  }
};

/**Function to view all Incident whose form is filled */
export const viewIncidentFilledForm = async (req, res) => {
  const { id } = req.query;
  const condition = await whereCondition({
    table: "incident",
    page: req.query.page,
    all: req.query.all,
    orderBy: req.query.orderBy,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    user: req.user,
  });

  const searchTableName = [
    "incident.location",
    "incident.description",
    "incident.current_impact",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
    "CONCAT(reporter_user.name , ' ' , reporter_user.surname)",
    "CONCAT(internal_user.name , ' ' , internal_user.surname)",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = incident.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = incident.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = incident.department",
    },
    {
      type: "left",
      targetTable: "users as reporter_user",
      onCondition: "reporter_user.id = incident.incident_reporting_by",
    },
    {
      type: "left",
      targetTable: "roles as reporter_role",
      onCondition: "reporter_user.role = reporter_role.id",
    },
    {
      type: "left",
      targetTable: "department as reporter_department",
      onCondition: "reporter_user.department = reporter_department.id",
    },
    {
      type: "left",
      targetTable: "permit_license_compliance",
      onCondition: "permit_license_compliance.id = incident.permit_license",
    },
    {
      type: "left",
      targetTable: "severity",
      onCondition: "severity.id = incident.initial_incident_severity",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all alert */
  let incidentDataFetchQuery = `SELECT 
  permit_license_compliance.name as permit_license_title,
  permit_license_compliance.end_date AS permit_license_expiry_date,
    incident.*,
    reporter_user.role,
    reporter_role.name as reporting_role_name,
    reporter_user.department,
    reporter_department.name as reporting_department,
    CONCAT(reporter_user.name , ' ' , reporter_user.surname) as incident_reporting_by_name,
    reporter_user.profile as incident_reporting_by_profile,
    severity.name as initial_incident_severity_name,
    CONCAT(users.name , ' ' , users.surname) as created_by_name,
    organization.name as organization_name,
    organization.header_image , organization.footer_image , organization.business_logo,
    department.name as department_name
    FROM incident  
    ${joinsRecord} 
    where incident.deleted = 0 ${searchCondition} ${condition}`;
  incidentDataFetchQuery = incidentDataFetchQuery.substring(
    0,
    incidentDataFetchQuery.indexOf("ORDER BY")
  );
  //incident.incident_reference_no,
  // console.log(incidentDataFetchQuery)
  let [incidentDataFetch] = await db.query(incidentDataFetchQuery);
  // return console.log(incidentDataFetch);
  /**Count all Incident */
  let arr = [];
  incidentDataFetch = await decodeAndParseFields(incidentDataFetch);
  for (const obj of incidentDataFetch) {
    if (obj.person_type == "Internal" || obj.person_type == "Both") {
      // console.log(obj?.internal_persons);
      const internal_persons_list = obj?.internal_persons;
      for (let i = 0; i < internal_persons_list?.length; i++) {
        const internal_person = internal_persons_list[i];
        // console.log(internal_person);
        const [fetch] = await db.query(
          `SELECT CONCAT(users.name , ' ' ,users.surname) AS name , roles.name AS role_name , department.name AS department_name FROM users LEFT JOIN roles ON users.role = roles.id LEFT JOIN department ON users.department = department.id WHERE users.id = ${internal_person?.user_id}`
        );
        /** get incident_role */
        let incidentRole = [];
        if (internal_person?.incident_role) {
          [incidentRole] = await db.query(
            `SELECT id as incident_role , name as incident_role_name FROM incident_role WHERE id = ${internal_person?.incident_role}`
          );
        }
        internal_persons_list[i] = {
          ...internal_person,
          ...fetch[0],
          ...incidentRole[0],
        };
      }
    }
    if (obj.person_type == "External" || obj.person_type == "Both") {
      const external_person_list = obj?.external_persons || [];
      for (let i = 0; i < external_person_list.length; i++) {
        const external_person = external_person_list[i];
        const is_contractor = external_person.is_contractor;
        if (is_contractor) {
          const [fetch] = await db.query(
            `SELECT contractor_name as contractor_name FROM contractor_registration WHERE id = ${external_person?.contractor_id}`
          );
          // console.log(fetch, "fetch");
          /** get incident_role */
          const [incidentRole] = await db.query(
            `SELECT id as incident_role , name as incident_role_name FROM incident_role WHERE id = ${external_person?.incident_role}`
          );

          /** get project registration also */
          const [project] = await db.query(
            `SELECT id as project_id , project_title as project_name FROM project_registration WHERE id = ${external_person?.project_id}`
          );
          external_person_list[i] = {
            // ...external_person,
            is_contractor: external_person.is_contractor,
            contractor_id: external_person.contractor_id,
            ...fetch[0],
            ...incidentRole[0],
            ...project[0],
          };
        } else {
          /** get incident_role */
          const [incidentRole] = await db.query(
            `SELECT id as incident_role , name as incident_role_name FROM incident_role WHERE id = ${external_person?.incident_role}`
          );

          /** get project registration also */
          const [project] = await db.query(
            `SELECT id as project_id , project_title as project_name FROM project_registration WHERE id = ${external_person?.project_id}`
          );
          external_person_list[i] = {
            // ...external_person,
            is_contractor: is_contractor,
            external_person: external_person.external_person,
            ...incidentRole[0],
            ...project[0],
          };
        }
      }
      obj.external_persons = external_person_list;
    }
    const id = obj.id;
    if (obj.incident_category && obj.incident_category.length > 0) {
      const [incidentCategoryList] = await db.query(
        `SELECT name from incident_category WHERE id In (${obj.incident_category}) AND deleted = 0`
      );

      obj.incident_category_name = incidentCategoryList.map(
        (item) => item?.name
      );
    }
    let usersList = [];
    if (obj.notification && obj.notification.length > 0) {
      [usersList] = await db.query(
        `SELECT users.id as user_id , CONCAT(users.name , ' ' , users.surname) as name , profile ,roles.id as role_id, roles.name as   role_name , 
        department.id as department_id, department.name as department_name from users
        LEFT JOIN roles ON roles.id = users.role 
        LEFT JOIN department ON department.id = users.department WHERE users.id In (${obj.notification}) AND users.deleted = "0"`
      );
    }

    // if (obj.is_done == 1) {
    // get investigation id of this incident
    const [record] = await getRecord("investigation", "incident_id", obj.id);
    if (record?.id) {
      obj.investigation_records = await getInvestigationDataIntoCompleted(
        record?.id
      );
    } else {
      obj.investigation_records = {};
    }

    obj.investigation_id = record?.id || "";
    obj.investigation_start_date = record?.investigation_start_date || "";
    obj.investigation_end_date = record?.investigation_end_date || "";
    obj.has_similar_incident_occurred_before =
      record?.has_similar_incident_occurred_before;
    obj.signature_type = record?.signature_type;
    obj.investigation_ddrm_id = record?.investigation_ddrm_id;
    // }
    // console.log(usersList, "usersList")
    obj.notification = usersList;

    /** check for all form submitted completely or not on a specific incident */

    const incidentCategoryQuery = `SELECT id, name , description FROM incident_category WHERE deleted = 0 AND id IN (${obj.incident_category})`;
    const [incidentCategoryList] = await db.query(incidentCategoryQuery);
    let result = [];
    let allFormSubmissionCheck = "Completed";
    let anyOneDraft = false;
    for (let i = 0; i < incidentCategoryList.length; i++) {
      const item = incidentCategoryList[i];
      if (incidentCategoryConstant[item?.name]) {
        item.description = await decodeSingle_statement(item.description);
        const incidentCategoryFormList = incidentCategoryConstant[item?.name];
        for (let j = 0; j < incidentCategoryFormList.length; j++) {
          const e = incidentCategoryFormList[j];
          /** Check the from is already added for this incident */
          let status = "Incomplete";
          let form_id = "";
          if (e.tableName) {
            const FormCheckQuery = `SELECT id, save_type FROM ${e.tableName} WHERE incident_id = ${id} AND deleted = 0 ORDER BY id DESC`;
            const [formRecord] = await db.query(FormCheckQuery);
            if (formRecord?.length > 0) {
              status = "Completed";
              form_id = formRecord[0].id;
            } else {
              form_id = "";
              allFormSubmissionCheck = "Incomplete";
            }
            // console.log(formRecord, "formRecord");
            if (formRecord[0]?.save_type == "draft") {
              status = "Draft";
              allFormSubmissionCheck = "Draft";
              anyOneDraft = true;
            }
          }

          result.push({
            ...item,
            title: e?.name,
            status: status,
            form_id: form_id,
            incident_id: id,
          });
        }
      }
    }
    obj.all_form_submission = anyOneDraft ? "Draft" : allFormSubmissionCheck;
    if (
      obj.all_form_submission !== "Draft" &&
      obj.all_form_submission !== "Incomplete"
    ) {
      arr.push(obj);
    }
    // console.log(obj);
    const imageDataQuery = `SELECT incident_docs.id, incident_docs.doc_no, incident_docs.incident_id, document_creation.document_name as title, incident_docs.ddrm_id, repository.url as file, file_classification.id as file_classification, file_classification.name as file_classification_name FROM incident_docs LEFT JOIN file_classification ON incident_docs.file_classification = file_classification.id LEFT JOIN repository ON incident_docs.ddrm_id = repository.id LEFT JOIN document_creation ON document_creation.id = repository.document_creation_id WHERE incident_docs.incident_id = ${id} AND incident_docs.deleted = 0`;
    const [imageDataFetch] = await db.query(imageDataQuery);
    // Assuming imageDataFetch contains an array of objects with image data
    obj.photographs_incident = imageDataFetch;

    if (obj?.incident_location?.length) {
      const [incidentLocation] = await db.query(
        `SELECT id,name from location WHERE id IN (${obj.incident_location}) AND deleted = 0`
      );
      obj.incident_location_name = incidentLocation;
    }
  }
  const pageSize = req.query.pageSize || 10;
  const totalRecords = arr.length;
  const currentPage = req.query.page || 1;
  const startIndex = (currentPage - 1) * pageSize;
  arr = arr.slice(startIndex, startIndex + pageSize);
  return res.status(200).json({
    data: arr,
    total: totalRecords,
  });
};

export const uploadSignatureInvestigation = async (req, res) => {
  // const { sidebar_id, id } = req.body;
  // const ddrm_id = await uploadToDDRM(sidebar_id, signature, req);

  const { query, values } = updateQueryBuilder(Investigation, {
    ...req.body,
  });
  const [result] = await db.query(query, values);
  console.log("result: ", result);
  if (result.affectedRows > 0) {
    return sendResponse(res, 200, "certificate uploaded successfully");
  } else {
    return sendResponse(res, 400, "Error uploading signature");
  }
};
