import db from "../db-config.js";
import {
  countQueryCondition,
  createQueryBuilder,
  decodeAndParseFields,
  deleteRecord,
  encodeAndStringifyFields,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";

import { sendResponse } from "../helper/wrapper.js";
import NcrRecording from "../sequelize/NcrRecordingSchema.js";

export const createUpdateNcrRecording = async (req, res) => {
  const { id, department } = req.body;
  let status = id ? "Updated" : "Created";
  req.body[id ? "updated_by" : "created_by"] = req.user.sessionid;
  if (department) {
    req.body.department = (
      await getOrganizationAccordingToDepartment(department)
    )[0].organization;
  }
  const unique_id = await uniqueIdGenerator(
    req.body.organization,
    department,
    "NCR",
    "ncr_recording",
    "unique_id",
    "unique_id"
  );
  const type = "reference";
  const reference_no = await uniqueIdGenerator(
    req.body.organization,
    department,
    "NCR",
    "ncr_recording",
    "reference_no",
    type
  );
  req.body.unique_id = unique_id;
  req.body.reference_no = reference_no;
  req.body = await encodeAndStringifyFields(req.body);
  const { query, values } = id
    ? updateQueryBuilder(NcrRecording, req.body)
    : createQueryBuilder(NcrRecording, req.body);
  const result = await db.query(query, values);
  await insertActivityLog(
    req.user.sessionid,
    status,
    "NCR Recording",
    id ? id : result[0].insertId
  );
  return sendResponse(res, 200, "Record " + status + " Successfully");
};

export const getNcrRecording = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "ncr_recording",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "ncr_recording.raised_by_employee_no",
    "ncr_recording.raised_by_contact",
    "ncr_recording.ncr_description",
    "ncr_recording.type_of_work",
    "ncr_recording.immediate_action_taken",
    "ncr_recording.legal_non_compliance",
    "ncr_recording.system_non_conformance",
    "ncr_recording.customer_non_compliance",
    "ncr_recording.name_of_legislation",
    "ncr_recording.name_of_standard",
    "ncr_recording.customer_name",
    "ncr_recording.ncr_title",
    // "ncr_recording.provision_requirement",
    "ncr_recording.standard_clause_name",
    "ncr_recording.contravened_contract_clause",
    "ncr_recording.classification",
    "ncr_recording.internal_name_of_site",
    "ncr_recording.internal_location",
    "ncr_recording.internal_employee_no",
    "ncr_recording.external_raised_person",
    "ncr_recording.external_contact_details",
    "ncr_recording.corrective_action_title",
    "ncr_recording.corrective_action_description",
    "ncr_recording.corrective_action_progress",
    "ncr_recording.corrective_status",
    "ncr_recording.preventative_action_title",
    "ncr_recording.preventative_action_description",
    "ncr_recording.preventative_action_progress",
    "ncr_recording.preventative_status",
    "ncr_recording.root_cause",
    "ncr_recording.type_of_resource",
    "ncr_recording.new_risk",
    "ncr_recording.problem_been_removed",
    "ncr_recording.comments",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = ncr_recording.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = ncr_recording.organization",
    },
    {
      type: "left",
      targetTable: "users as raisedUser",
      onCondition: "raisedUser.id = ncr_recording.raised_by",
    },
    {
      type: "left",
      targetTable: "organization AS internalRaisedOrg",
      onCondition:
        "internalRaisedOrg.id = ncr_recording.internal_raised_business",
    },
    {
      type: "left",
      targetTable: "organization AS internalAgainstOrganization",
      onCondition:
        "internalAgainstOrganization.id = ncr_recording.internal_raised_against_business",
    },
    {
      type: "left",
      targetTable: "department AS internalAgainstDepartment",
      onCondition:
        "internalAgainstDepartment.id = ncr_recording.internal_raised_against_department",
    },
    {
      type: "left",
      targetTable: "users AS internalResponsible",
      onCondition:
        "internalResponsible.id = ncr_recording.internal_responsible_person",
    },
    {
      type: "left",
      targetTable: "users AS correctiveResponsible",
      onCondition:
        "correctiveResponsible.id = ncr_recording.corrective_responsible_person",
    },
    {
      type: "left",
      targetTable: "department AS correctiveDepartment",
      onCondition:
        "correctiveDepartment.id = ncr_recording.corrective_responsible_department",
    },
    {
      type: "left",
      targetTable: "department AS preventiveDepartment",
      onCondition:
        "preventiveDepartment.id = ncr_recording.preventative_responsible_department",
    },
    {
      type: "left",
      targetTable: "users AS preventiveUser",
      onCondition:
        "preventiveUser.id = ncr_recording.preventative_responsible_person",
    },
    {
      type: "left",
      targetTable: "name_of_standard",
      onCondition: "name_of_standard.id = ncr_recording.name_of_standard",
    },
    {
      type: "left",
      targetTable: "repository as  stop_certification",
      onCondition:
        "stop_certification.id = ncr_recording.stop_certificate_ddrm_id",
    },
    {
      type: "left",
      targetTable: "repository as ncr_report",
      onCondition: "ncr_report.id = ncr_recording.ncr_report_ddrm_id",
    },
    {
      type: "left",
      targetTable: "users as recipient_user",
      onCondition: "recipient_user.id = ncr_recording.recipient_user_id",
    },
    {
      type: "left",
      targetTable: "users as issuer_user",
      onCondition: "issuer_user.id = ncr_recording.issuer_user_id",
    },
    {
      type: "left",
      targetTable: "location",
      onCondition: "location.id = ncr_recording.internal_location",
    },
    {
      type: "left",
      targetTable: "currency",
      onCondition: "currency.id = ncr_recording.currency",
    },
  ];

  const joinCondition = await makeJoins(joins);

  const ncrRecordingQuery = `SELECT ncr_recording.* , currency.name as currency_name, stop_certification.url as stop_certificate_file, 
  ncr_report.url as ncr_report_file, location.name as internal_location_name, CONCAT(recipient_user.name , ' ' , recipient_user.surname) AS recipient_user_name, recipient_user.profile AS recipient_user_profile, CONCAT(issuer_user.name , ' ' , issuer_user.surname) AS issuer_user_name, issuer_user.profile AS issuer_user_profile, users.name as created_by_name,users.surname as created_by_surname, users.profile as created_by_profile ,organization.name as organization_name , CONCAT(raisedUser.name , ' ' , raisedUser.surname) AS raised_by_name , raisedUser.profile AS raised_by_profile , raisedUser.unique_id as raised_by_employee_no , raisedUser.phone AS raised_by_contact , internalRaisedOrg.name AS internal_raised_business_name , internalAgainstOrganization.name AS internal_raised_against_business_name , internalAgainstDepartment.name AS internal_raised_against_department_name , CONCAT(internalResponsible.name , ' ' , internalResponsible.surname) AS internal_responsible_person_name , internalResponsible.unique_id AS internal_employee_no , internalResponsible.profile AS internal_responsible_person_profile , CONCAT(correctiveResponsible.name , ' ' , correctiveResponsible.surname) as corrective_responsible_person_name ,correctiveResponsible.profile as corrective_responsible_person_profile , correctiveDepartment.name AS corrective_responsible_department_name , preventiveDepartment.name AS preventative_responsible_department_name , organization.header_image , organization.footer_image , organization.business_logo, CONCAT(preventiveUser.name , ' ' , preventiveUser.surname) AS preventative_responsible_person_name , preventiveUser.profile AS preventative_responsible_person_profile , name_of_standard.name AS name_of_standard_name FROM ncr_recording ${joinCondition} WHERE ncr_recording.deleted = 0 ${searchCondition} ${condition}`;
  // console.log(ncrRecordingQuery);
  let [ncrRecording] = await db.query(ncrRecordingQuery);
  ncrRecording = await decodeAndParseFields(ncrRecording);
  for (const i of ncrRecording) {
    const categories = i.ncr_categories;
    let cat = [];
    for (const category of categories) {
      const [name] = await db.query(
        `SELECT name FROM incident_category WHERE id = ${category}`
      );
      cat.push(name[0].name);
    }
    i.ncr_categories_name = cat;
    const requirement = i.human_resource_requirements;
    for (const r of requirement) {
      // console.log('r: ', r);
      r.role = r.role || null;
      let name;
      if (r.role) {
        [name] = await db.query(`SELECT name FROM roles WHERE id = ${r.role}`);
      }
      // console.log('name', name)
      r.role_name = name && name[0].name;
    }
  }

  const totalRecord = await countQueryCondition(ncrRecordingQuery);

  return sendResponse(res, 200, ncrRecording, totalRecord);
};

export const deleteNcrRecording = async (req, res) => {
  const { id } = req.params;
  await deleteRecord(NcrRecording, id);

  /**Insert record for activity log */
  await insertActivityLog(req.user.sessionid, "delete", "NCR Recording", id);
  return sendResponse(res, 200, "Record deleted successfully");
};

export const uploadSignatureNCR = async (req, res) => {
  // const { sidebar_id, id } = req.body;
  // const ddrm_id = await uploadToDDRM(sidebar_id, signature, req);
  console.log(req.body, "req.body");
  const { query, values } = updateQueryBuilder(NcrRecording, {
    ...req.body,
  });
  const [result] = await db.query(query, values);
  if (result.affectedRows > 0) {
    return sendResponse(res, 200, "certificate uploaded successfully");
  } else {
    return sendResponse(res, 400, "Error uploading signature");
  }
};
