import {
  countQueryCondition,
  decodeAndParseFields,
  decodeSingle_statement,
  deleteRecord,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  getRecord,
  insertActivityLog,
  makeJoins,
  searchConditionRecord,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import OccupationalDiseaseForm from "../sequelize/OccupationalDiseaseTechnicalFormSchema.js";
import db from "../db-config.js";
import { createQueryBuilder } from "../helper/queryBuilder.js";
import { sendResponse } from "../helper/wrapper.js";
import moment from "moment";

/**Function to create Safety Incident Form  */
export const createOccupationalDiseaseForm = async (req, res) => {
  const { id } = req.body;
  const sessionId = req.user.sessionid;
  let organizationId = req.body.organization;
  const { department, comments, description, names_of_exposed_employees } =
    req.body;

  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0]?.organization;
    if (organizationId) {
      req.body.organization = organizationId;
    }
  }
  req.body.names_of_exposed_employees = JSON.stringify(
    names_of_exposed_employees
  );
  req.body.comments = await encodeSingle_statement(comments);
  req.body.description = await encodeSingle_statement(description);
  req.body.created_by = sessionId;
  req.body.incident_category = JSON.stringify(req.body.incident_category);
  req.body.incident_date = moment(req.body.incident_date).format("YYYY-MM-DD");
  // console.log('req.body: ', req.body);
  const { query, values } = id
    ? updateQueryBuilder(OccupationalDiseaseForm, req.body)
    : createQueryBuilder(OccupationalDiseaseForm, req.body);
  const [createOccupationalDiseaseForm] = await db.query(query, values);
  if (!id && createOccupationalDiseaseForm.insertId === 0) {
    return sendResponse(res, 404, "Cannot create record");
  }

  await insertActivityLog(
    req.user.sessionid,
    "create",
    "Occupational Disease Form",
    `This user created a new Occupational Disease Form Record for organization ${organizationId}`
  );

  return sendResponse(res, 200, "Record created successfully");
};

export const getOccupationalDiseaseForm = async (req, res) => {
  const { type } = req.query;
  const { id } = req.params;
  const condition = await whereCondition({
    table: "occupational_disease_form",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
    columnName: type,
  });
  const searchTableName = [
    "occupational_disease_form.occupational_disease_details",
    "occupational_disease_form.roles.name",
    "occupational_disease_form.were_they_licensed_trained",
    "occupational_disease_form.was_license_training_applicable_to_task",
    "occupational_disease_form.license_no_training_certificate_no",
    "occupational_disease_form.year",
    "occupational_disease_form.issued_by",
    "occupational_disease_form.occupational_disease_category",
    "occupational_disease_form.irreversible_occupational_disease",
    "occupational_disease_form.noise_induced_hearing_loss",
    "occupational_disease_form.description_of_occupational_disease_illness",
    "occupational_disease_form.was_case_reported_to_authority_regulator",
    "occupational_disease_form.name_of_regulator",
    "occupational_disease_form.reported_by_who",
    "occupational_disease_form.effect_on_person",
    "occupational_disease_form.exposure_route",
    "occupational_disease_form.source_of_exposure",
    "occupational_disease_form.estimated_duration_of_exposure",
    "occupational_disease_form.other_employees_exposed",
    "occupational_disease_form.names_of_exposed_employees",
    "occupational_disease_form.occupational_hygiene_survey_conducted",
    "occupational_disease_form.comments",
    "occupational_disease_form.form_completed_by",
    "occupational_disease_form.designation",
    "occupational_disease_form.signature",
    "occupational_disease_form.department.name",
    "occupational_disease_form.organization.name",
    "occupational_disease_form.CONCAT(users.name , ' ' , users.surname)",
  ];
  const searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = occupational_disease_form.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = occupational_disease_form.organization",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = occupational_disease_form.department",
    },
    {
      type: "left",
      targetTable: "category",
      onCondition: "category.id = occupational_disease_form.category_id",
    },
    {
      type: "left",
      targetTable: "users AS exposed",
      onCondition:
        "exposed.id = occupational_disease_form.name_of_exposed_person",
    },
    {
      type: "left",
      targetTable: "roles",
      onCondition: "roles.id = exposed.role",
    },
  ];
  const joinCondition = await makeJoins(joins);
  const occupationalDiseaseFormQuery = `SELECT occupational_disease_form.* , organization.name as organization_name ,department.name as department_name , users.name as created_by_name , users.surname as created_by_surname  , users.profile as created_by_profile , category.name as category_name , roles.name as role FROM occupational_disease_form ${joinCondition} WHERE occupational_disease_form.deleted = 0 ${condition} ${searchCondition}`;
  const totalRecord = await countQueryCondition(occupationalDiseaseFormQuery);

  let [occupationalDiseaseForm] = await db.query(occupationalDiseaseFormQuery);
  occupationalDiseaseForm = await decodeAndParseFields(occupationalDiseaseForm);
  return sendResponse(res, 200, occupationalDiseaseForm, totalRecord);
};

export const deleteOccupationalDiseaseForm = async (req, res) => {
  const { id } = req.params;
  const deleteForm = await deleteRecord("occupational_disease_form", id);
  if (deleteForm) {
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Occupational Disease Form",
      `This user deleted a new Occupational disease Form Record for organization ${id}`
    );
    return sendResponse(res, 200, "Record deleted successfully");
  }
  return sendResponse(res, 404, "Cannot delete record");
};
