import asyncHandler from "express-async-handler";
import db from "../db-config.js";
import {
  insertActivityLog,
  storeError,
  whereCondition,
  makeJoins,
  countQueryCondition,
  deleteRecord,
  getOrganizationAccordingToDepartment,
  encodeSingle_statement,
  decodeSingle_statement,
  decodeTheEditorContent,
  encodeTheEditorContent,
  searchConditionRecord,
  updateQueryBuilder,
  createQueryBuilder,
  decodeAndParseFields,
  getRecord,
  getOperationalRecordArray,
  sendDynamicEmail,
  getOperationalRiskData,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import OperationalRiskIdentification from "../sequelize/OperationalRiskIdentificationSchema.js";
import OperationalRisk from "../sequelize/OperationalRiskSchema.js";

/**function to create and update operational risk identification  */
export const createUpdateOperationalRiskIdentification = async (req, res) => {
  const {
    id,
    hazard_description,
    risk_description,
    impact_description,
    control_description,
    opportunity_description,
    organization,
    department,
  } = req.body;

  req.body.hazard_description = await encodeSingle_statement(
    hazard_description
  );
  req.body.control_description = await encodeSingle_statement(
    control_description
  );
  req.body.impact_description = await encodeSingle_statement(
    impact_description
  );
  req.body.risk_description = await encodeSingle_statement(risk_description);
  req.body.opportunity_description =
    opportunity_description.length > 0
      ? JSON.stringify(
          await encodeTheEditorContent(opportunity_description, "description")
        )
      : "[{}]";

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    /**Update Operational Risk Identification Query */
    const { query, values } = updateQueryBuilder(
      OperationalRiskIdentification,
      req.body
    );
    await db.query(query, values);

    /**Insert Activity  */
    await insertActivityLog(
      req.user.sessionid,
      "update",
      "Operational Risk Identification",
      id
    );

    return sendResponse(res, 200, "Record updated Successfully");
  } else {
    req.body.created_by = req.user.sessionid;
    const { query, values } = createQueryBuilder(
      OperationalRiskIdentification,
      req.body
    );

    const [createOperationalRiskIdentification] = await db.query(query, values);
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Operational Risk Identification",
      createOperationalRiskIdentification.insertId
    );

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single operational risk identification */
export const getOperationalRiskIdentification = async (req, res) => {
  const { id } = req.params;
  const { status, approval_status } = req.query;
  const condition = await whereCondition({
    table: "operational_risk_identification",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const statusFilter = status
    ? `AND operational_risk.register_status = '${status}'`
    : "";
  const approvalFilter = approval_status
    ? `AND operational_risk.approval_status = '${approval_status}'`
    : "";
  const searchTableName = [
    "CONCAT(users.name , ' ' , users.surname)",
    "operational_risk.unique_id",
    "operational_risk.risk_register_name",
    "operational_risk.risk_context",
    "operational_risk.register_status",
    "operational_risk_identification.name",
    "operational_risk_identification.residual_rating",
    "operational_risk_identification.residual_ranking",
    "operational_risk_identification.opportunity_description",
    "operational_risk_identification.hazard",
    "operational_risk_identification.description",
    "operational_risk_identification.impact",
    "category.name",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = operational_risk_identification.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = operational_risk_identification.organization",
    },
    {
      type: "left",
      targetTable: "operational_activities",
      onCondition:
        "operational_risk_identification.activity_id = operational_activities.id",
    },
    {
      type: "left",
      targetTable: "operational_work_area",
      onCondition: "operational_work_area.id = operational_activities.work_id",
    },
    {
      type: "left",
      targetTable: "operational_risk",
      onCondition:
        "operational_risk.id = operational_work_area.operational_risk_id",
    },
    {
      type: "left",
      targetTable: "category",
      onCondition: "category.id = operational_risk_identification.category",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = operational_risk.meeting_id",
    },
    // {
    //   type: "left",
    //   targetTable: "users as riskOwner",
    //   onCondition: "riskOwner.id = operational_risk_identification.risk_owner",
    // },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all operational risk */
  const operationalRiskQuery = `SELECT operational_risk_identification.* , operational_risk_identification.name as risk_name, operational_work_area.operational_risk_id, operational_risk.risk_register_name , operational_risk.unique_id, operational_risk.assessment_date, operational_risk.review_date, operational_risk.main_process, meeting.meeting_title as meeting_name,  operational_risk.risk_context, operational_risk.register_status AS status ,organization.name as organization_name,users.name as created_by , operational_risk.risk_register_name, operational_risk.approval_status , category.name as category_name, operational_risk.status,operational_risk.register_status ,users.surname as created_by_surname,users.profile as created_by_profile
      FROM operational_risk_identification 
      ${joinsRecord} where operational_risk_identification.deleted = 0 ${statusFilter} ${approvalFilter} ${searchCondition} ${condition}`;

  // Modify the ORDER BY clause to ASC
  // console.log('operationalRiskQuery: ', operationalRiskQuery);
  // let query = operationalRiskQuery.replace(/ORDER BY .*? DESC/, 'ORDER BY operational_risk_identification.residual_ranking ASC');
  let query = operationalRiskQuery.replace(
    /ORDER BY .*? DESC/,
    `ORDER BY FIELD(operational_risk_identification.residual_ranking, 'Low', 'Medium', 'High', 'Very High') ASC`
  );

  let [operationalRisk] = await db.query(query);

  /**Decode content */
  operationalRisk = await decodeAndParseFields(operationalRisk);

  operationalRisk = operationalRisk.map((record) => {
    return {
      ...record,
      opportunity_description: JSON.parse(record.opportunity_description),
    };
  });

  const totalRecord = await countQueryCondition(operationalRiskQuery);

  return sendResponse(res, 200, operationalRisk, totalRecord);
};

export const updateOperationalRiskIdentification = async (req, res) => {
  const { id, priority_confirmation } = req.body;

  if (!id || !priority_confirmation) {
    return sendResponse(
      res,
      400,
      "Please provide id and priority_confirmation"
    );
  }

  let [riskIdentification] = await db.query(
    `SELECT risk_name, risk_register_id, risk_description, CONCAT(users.name , ' ' , users.surname) as risk_owner_name, users.email as risk_owner_email FROM operational_risk_identification LEFT JOIN users ON users.id = operational_risk_identification.risk_owner WHERE operational_risk_identification.id = ? AND operational_risk_identification.deleted = 0`,
    [id]
  );

  riskIdentification = await decodeAndParseFields(riskIdentification);

  const additionalData = {
    risk_name: riskIdentification[0].risk_name,
    risk_description: riskIdentification[0].risk_description,
  };

  const [operationalData] = await getOperationalRiskData(
    riskIdentification[0].risk_register_id
  );
  const templateFileUrl = "mail_for_operational_risk_prioritised_template.html";
  const templateName = "Operational Risk Prioritised";
  const dataToSend = await getOperationalRecordArray({
    operationalData,
    templateFileUrl,
    templateName,
  });
  await sendDynamicEmail({
    to: operationalData.approver_email,
    subject: `Priority Risk Notification - ${additionalData.risk_name}`,
    data: { ...dataToSend, ...additionalData },
  });
  const { query, values } = updateQueryBuilder(
    OperationalRiskIdentification,
    req.body
  );
  const [updatePriority] = await db.query(query, values);

  if (updatePriority.affectedRows > 0) {
    return sendResponse(res, 200, "Record updated successfully");
  } else {
    return sendResponse(res, 400, "Record not updated");
  }
};

/**Function to view all and single operational risk identification */
export const getOperationalRiskIdentificationApprovalWorkflowOld = async (
  req,
  res
) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "operational_risk_identification",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "CONCAT(users.name , ' ' , users.surname)",
    "operational_risk.unique_id",
    "operational_risk.risk_register_name",
    "operational_risk.risk_context",
    "operational_risk.register_status",
    "operational_risk_identification.name",
    "operational_risk_identification.residual_rating",
    "operational_risk_identification.residual_ranking",
    "operational_risk_identification.opportunity_description",
    "operational_risk_identification.hazard",
    "operational_risk_identification.description",
    "operational_risk_identification.impact",
    "category.name",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = operational_risk_identification.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = operational_risk_identification.organization",
    },
    {
      type: "left",
      targetTable: "operational_activities",
      onCondition:
        "operational_risk_identification.activity_id = operational_activities.id",
    },
    {
      type: "left",
      targetTable: "operational_work_area",
      onCondition: "operational_work_area.id = operational_activities.work_id",
    },
    {
      type: "left",
      targetTable: "operational_risk",
      onCondition:
        "operational_risk.id = operational_work_area.operational_risk_id",
    },
    // {
    //   type: "left",
    //   targetTable: "users as riskOwner",
    //   onCondition: "riskOwner.id = operational_risk_identification.risk_owner",
    // },
    // {
    //   type: "left",
    //   targetTable: "category_type",
    //   onCondition:
    //     "category_type.id = operational_risk_identification.category_description",
    // },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all operational risk */
  const operationalRiskQuery = `SELECT operational_risk_identification.* , operational_work_area.operational_risk_id, operational_risk.risk_register_name , operational_risk.risk_context, operational_risk.register_status AS status ,organization.name as organization_name,users.name as created_by , operational_risk.risk_register_name, operational_risk.approval_status , operational_risk.status,operational_risk.register_status ,users.surname as created_by_surname,users.profile as created_by_profile
      FROM operational_risk_identification 
      ${joinsRecord} where operational_risk_identification.deleted = 0  AND operational_risk.register_status = 'Complete' AND (operational_risk.approval_status != 'Approved'  AND  operational_risk.approval_status != 'Rejected')   ${searchCondition} ${condition}`;
  let [operationalRisk] = await db.query(operationalRiskQuery);

  /**Decode content */
  operationalRisk = await decodeAndParseFields(operationalRisk);

  operationalRisk = operationalRisk.map((record) => {
    return {
      ...record,
      opportunity_description: JSON.parse(record.opportunity_description),
    };
  });

  const totalRecord = await countQueryCondition(operationalRiskQuery);

  return sendResponse(res, 200, operationalRisk, totalRecord);
};

// export const getOperationalRiskIdentificationApprovalWorkflow = async (
//   req,
//   res
// ) => {
//   const { id } = req.params;

//   const condition = await whereCondition({
//     table: "operational_risk",
//     page: req.query.page,
//     all: req.query.all,
//     pageSize: req.query.pageSize,
//     filter: req.query.filter,
//     id,
//     grouped: req.query.grouped,
//     user: req.user,
//   });

//   const searchTableName = [
//     "CONCAT(facilitator.name , ' ' , facilitator.surname)",
//     "CONCAT(users.name , ' ' , users.surname)",
//     "CONCAT(approver.name , ' ' , approver.surname)",
//     "operational_risk.risk_register_name",
//     "operational_risk.risk_context",
//     "organization.name",
//   ];

//   const searchCondition = await searchConditionRecord(
//     req.query.search,
//     searchTableName
//   );

//   const joins = [
//     {
//       type: "left",
//       targetTable: "users",
//       onCondition: "users.id = operational_risk.created_by",
//     },
//     {
//       type: "left",
//       targetTable: "organization",
//       onCondition: "organization.id = operational_risk.organization",
//     },
//     {
//       type: "left",
//       targetTable: "meeting",
//       onCondition: "meeting.id = operational_risk.meeting_id",
//     },
//     {
//       type: "left",
//       targetTable: "users as facilitator",
//       onCondition:
//         "facilitator.id = operational_risk.risk_assessment_facilitator",
//     },
//     {
//       type: "left",
//       targetTable: "users as approver",
//       onCondition: "approver.id = operational_risk.risk_register_approver",
//     },
//   ];

//   const joinsRecord = await makeJoins(joins);

//   const operationalRiskQuery = `SELECT operational_risk.*,
//     organization.name as organization_name, users.name as created_by, users.surname as created_by_surname,
//     facilitator.name as facilitator_name, facilitator.surname as facilitator_surname,
//     approver.name as approver_name, approver.surname as approver_surname,
//     meeting.meeting_title as meeting_name
//     FROM operational_risk
//     ${joinsRecord} WHERE operational_risk.deleted = 0  AND operational_risk.register_status = 'Complete' AND (operational_risk.approval_status != 'Approved'  AND  operational_risk.approval_status != 'Rejected')  ${searchCondition} ${condition}`;
//   let [operationalRisk] = await db.query(operationalRiskQuery);

//   operationalRisk = await decodeAndParseFields(operationalRisk);

//   // Prepare the structured data
//   for (let risk of operationalRisk) {
//     // Fetch associated work areas
//     const [workAreas] = await db.query(
//       `SELECT * FROM operational_work_area WHERE operational_risk_id = ${risk.id} AND deleted = 0`
//     );
//     risk.work_areas = await Promise.all(
//       workAreas.map(async (workArea) => {
//         const [activities] = await db.query(
//           `SELECT * FROM operational_activities WHERE work_id = ${workArea.id} AND deleted = 0`
//         );
//         workArea.activities = await Promise.all(
//           activities.map(async (activity) => {
//             const [risks] = await db.query(
//               `SELECT operational_risk_identification.*, category_type.name as category_name FROM operational_risk_identification  LEFT JOIN category_type ON category_type.id = operational_risk_identification.category WHERE operational_risk_identification.activity_id = ${activity.id} AND operational_risk_identification.deleted = 0`
//             );
//             activity.risks = risks.map((riskData) => ({
//               ...riskData,
//               control_assessments: JSON.parse(
//                 riskData.control_assessments || "[]"
//               ),
//               opportunity_description: JSON.parse(
//                 riskData.opportunity_description || "[]"
//               ),
//             }));
//             return {
//               ...activity,
//               risks: await decodeAndParseFields(activity.risks),
//             };
//           })
//         );
//         return {
//           ...workArea,
//           activities: await decodeAndParseFields(workArea.activities),
//         };
//       })
//     );
//   }

//   const totalRecord = await countQueryCondition(operationalRiskQuery);
//   return sendResponse(res, 200, operationalRisk, totalRecord);
// };

export const getOperationalRiskIdentificationApprovalWorkflow = async (
  req,
  res
) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "operational_risk_identification",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });
  const { status } = req.query;
  const statusFilter = status
    ? `AND operational_risk.register_status = '${status}'`
    : "";
  const searchTableName = [
    "CONCAT(users.name , ' ' , users.surname)",
    "operational_risk.unique_id",
    "operational_risk.risk_register_name",
    "operational_risk.risk_context",
    "operational_risk.register_status",
    "operational_risk_identification.name",
    "operational_risk_identification.residual_rating",
    "operational_risk_identification.residual_ranking",
    "operational_risk_identification.opportunity_description",
    "operational_risk_identification.hazard",
    "operational_risk_identification.description",
    "operational_risk_identification.impact",
    "category.name",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = operational_risk_identification.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition:
        "organization.id = operational_risk_identification.organization",
    },
    {
      type: "left",
      targetTable: "operational_activities",
      onCondition:
        "operational_risk_identification.activity_id = operational_activities.id",
    },
    {
      type: "left",
      targetTable: "operational_work_area",
      onCondition: "operational_work_area.id = operational_activities.work_id",
    },
    {
      type: "left",
      targetTable: "operational_risk",
      onCondition:
        "operational_risk.id = operational_work_area.operational_risk_id",
    },
    {
      type: "left",
      targetTable: "category",
      onCondition: "category.id = operational_risk_identification.category",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = operational_risk.meeting_id",
    },
    // {
    //   type: "left",
    //   targetTable: "users as riskOwner",
    //   onCondition: "riskOwner.id = operational_risk_identification.risk_owner",
    // },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all operational risk */
  const operationalRiskQuery = `SELECT operational_risk_identification.* , operational_risk_identification.name as risk_name, operational_work_area.operational_risk_id, operational_risk.risk_register_name , operational_risk.unique_id, operational_risk.assessment_date, operational_risk.review_date, operational_risk.main_process, meeting.meeting_title as meeting_name,  operational_risk.risk_context, operational_risk.register_status AS status ,organization.name as organization_name,users.name as created_by , operational_risk.risk_register_name, operational_risk.approval_status , category.name as category_name, operational_risk.status,operational_risk.register_status ,users.surname as created_by_surname,users.profile as created_by_profile
      FROM operational_risk_identification 
      ${joinsRecord} where operational_risk_identification.deleted = 0 AND operational_risk.register_status = 'Complete' AND (operational_risk.approval_status != 'Approved'  AND  operational_risk.approval_status != 'Rejected') ${statusFilter} ${searchCondition} ${condition}`;

  // Modify the ORDER BY clause to ASC
  // console.log('operationalRiskQuery: ', operationalRiskQuery);
  // let query = operationalRiskQuery.replace(/ORDER BY .*? DESC/, 'ORDER BY operational_risk_identification.residual_ranking ASC');
  let query = operationalRiskQuery.replace(
    /ORDER BY .*? DESC/,
    `ORDER BY FIELD(operational_risk_identification.residual_ranking, 'Low', 'Medium', 'High', 'Very High') ASC`
  );

  let [operationalRisk] = await db.query(query);

  /**Decode content */
  operationalRisk = await decodeAndParseFields(operationalRisk);

  operationalRisk = operationalRisk.map((record) => {
    return {
      ...record,
      opportunity_description: JSON.parse(record.opportunity_description),
    };
  });

  const totalRecord = await countQueryCondition(operationalRiskQuery);

  return sendResponse(res, 200, operationalRisk, totalRecord);
};

/**Function to delete a specific Operational risk identification */
export const deleteOperationalRiskIdentification = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord(
    "operational_risk_identification",
    id
  );
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "delete",
      "Operational Risk Identification",
      id
    );
    return sendResponse(res, 200, "Record deleted Successfully");
  }
};

/**Function to approve and reject Operational risk identification */
export const approveRejectOperationalRiskIdentification = async (req, res) => {
  const { operationalRiskIdentificationId, approval_status } = req.body;
  let messageStatus = approval_status === "Approved" ? "approved" : "reject";
  req.body.id = operationalRiskIdentificationId;
  req.body.updated_by = req.user.sessionid;
  const { query, values } = updateQueryBuilder(OperationalRisk, req.body);
  await db.query(query, values);

  /**Insert record for activity log */
  await insertActivityLog(
    req.user.sessionid,
    messageStatus,
    "Operational Risk Identification",
    operationalRiskIdentificationId
  );
  return sendResponse(res, 200, `Record ${messageStatus} Successfully`);
};
