import path from "path";
import {
  auditLogFunction,
  extractSubdomain,
  getDomain,
  getFolderTotalSizeRecursively,
  getOrganizationAccordingToDepartment,
  getPathAfterHost,
  getUniqueFolderName,
  insertActivityLog,
  makeJoins,
  whereCondition,
} from "../helper/general.js";
import {
  createQueryBuilder,
  updateQueryBuilder,
} from "../helper/queryBuilder.js";
import { sendResponse } from "../helper/wrapper.js";
import Repository from "../sequelize/RepositorySchema.js";
import fs from "fs";
import chalk from "chalk";
import db from "../db-config.js";
import { fileURLToPath, URL } from "url";
import DocumentCreation from "../sequelize/DocumentCreationSchema.js";
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

export const createFolder = async (req, res) => {
  // const __dirname = path.dirname(new URL(import.meta.url).pathname);
  // const correctedDirname = __dirname.startsWith("/")
  //   ? __dirname.slice(1)
  //   : __dirname;
  let { folderName, parentId, organization, department, type } = req.body;

  if (!folderName) {
    return sendResponse(res, 400, "folderName is required");
  }

  if (folderName === "root") {
    return sendResponse(
      res,
      400,
      "cannot create root folder: Permission denied"
    );
  }

  if (department) {
    organization = (await getOrganizationAccordingToDepartment(department))[0]
      .organization;
  }

  let parentPath = "";
  if (parentId) {
    const [parentData] = await db.query(
      `SELECT * FROM repository WHERE id = '${parentId}' AND organization = '${organization}' AND deleted = 0`
    );
    if (!parentData[0]) {
      return sendResponse(res, 400, "Parent folder does not exist");
    }
    parentPath = parentData[0]?.url;
  }

  let parentCondition = ` parent_id = ${parentId}`;
  if (parentId == null) {
    parentCondition = " parent_id IS NULL";
  }
  const [siblings] = await db.query(
    `SELECT * FROM repository WHERE ${parentCondition} AND organization = ${organization} AND deleted = 0`
  );

  // Get unique folder name
  const publicPath = path.join(process.cwd(), "public");
  const uniqueFolderName = getUniqueFolderName(siblings, folderName);
  const absolutePath = path.join(publicPath, parentPath, uniqueFolderName);
  if (!fs.existsSync(absolutePath)) {
    fs.mkdirSync(absolutePath, { recursive: true });
    console.log(chalk.yellow(`Created folder: ${absolutePath}`));
  } else {
    console.log(chalk.red(`Folder already exists: ${absolutePath}`));
    return sendResponse(res, 400, "Folder already exists");
  }

  // const relativePath = `${parentPath}/${uniqueFolderName}`;
  const relativePath = path.join(parentPath, uniqueFolderName);

  // const [createdData] = await db.query(
  //   `INSERT INTO repository (name, organization, type, url, parent_id) VALUES ('${uniqueFolderName}', ${organization}, 'folder', '${relativePath}', ${parentId})`
  // );
  const { query, values } = createQueryBuilder(Repository, {
    name: uniqueFolderName,
    url: relativePath,
    parent_id: parentId,
    organization,
    type,
    created_by: req.user.sessionid,
  });
  const [createdData] = await db.query(query, values);

  // await insertActivityLog(
  //   req.user.sessionid,
  //   "create",
  //   "Repository",
  //   `This user created Repository with id: ${createdData?.id}`
  // );
  // let currentPath = path.join(correctedDirname, `../public/${folderName}`);
  // let currentPath = path.join(__dirname, `../public/ddrm/${folderName}`);
  // console.log("currentPath: ", currentPath);

  // if (parentId != 0) {
  //   let [parentData] = await db.query(`SELECT * FROM repository WHERE id = ${parentId} AND deleted = 0`);
  //   // console.log(parentData[0]);
  //   currentPath = path.join(parentData[0]?.url, `${folderName}`);
  //   console.log(chalk.yellow(currentPath));
  //   // return sendResponse(res, 400, "Folder is not parent");
  // }
  // if (!fs.existsSync(currentPath)) {
  //   // Create the folder recursively
  //   fs.mkdirSync(currentPath, { recursive: true });
  //   console.log(chalk.yellow(`Created folder: ${currentPath}`));
  // } else {
  //   console.log(chalk.red(`Folder already exists: ${currentPath}`));
  //   return sendResponse(res, 400, "Folder already exists");
  // }
  // console.log(currentPath.split("public"), "");
  // let filePath = currentPath.split("public")[1] ? currentPath.split("public")[1] : currentPath.split("public")[0];
  // const { query, values } = createQueryBuilder(Repository, {
  //   name: folderName,
  //   url: filePath,
  //   parent_id: parentId,
  //   organization,
  //   type,
  //   created_by: req.user.sessionid,
  // });
  // const [createdData] = await db.query(query, values);

  await insertActivityLog(
    req.user.sessionid,
    "create",
    "Repository",
    `This user created Repository with id: ${createdData?.id}`
  );

  return sendResponse(res, 200, "Folder created successfully");
};

export const getFolderByParent = async (req, res) => {
  let { id } = req.params;

  let parentCondition = "";
  if (id == "null") {
    parentCondition = " repository.parent_id IS NULL";
  } else {
    parentCondition = ` repository.parent_id = ${id}`;
  }
  const condition = await whereCondition({
    table: "repository",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    user: req.user,
    grouped: req.query.grouped,
  });

  const joins = [
    {
      type: "left",
      targetTable: "document_creation",
      onCondition: "document_creation.id = repository.document_creation_id",
    },
  ];
  const joinsRecord = await makeJoins(joins);
  const viewingRightCondition = req.user.isSuperAdmin
    ? ""
    : `AND (
    (
      repository.type = 'file'
      AND JSON_VALID(document_creation.viewing_rights_employees) 
      AND JSON_CONTAINS(document_creation.viewing_rights_employees, '${req.user.sessionid}')
    )
      OR (
      repository.type = 'folder'
      )
    )`;

  const getFolderData = `SELECT repository.* FROM repository ${joinsRecord} WHERE repository.deleted = 0 AND repository.is_archived = 0  AND ${parentCondition} ${viewingRightCondition} ${condition}`;

  const [folderData] = await db.query(getFolderData);

  // let totalSize = 0;
  if (
    folderData.length > 0 &&
    folderData.some((folder) => folder.type === "folder")
  ) {
    const finalData = [];

    // Process each folder in folderData
    for (const folder of folderData) {
      // console.log("folder: ", folder);
      if (folder.type === "folder" && folder.url) {
        // Calculate the total size recursively for this folder
        const totalSize = getFolderTotalSizeRecursively(
          folder.type,
          folder.url
        );
        // console.log(`Size for folder '${folder.url}': `, totalSize);
        console.log(
          `Size for folder '${folder.url}': ` +
            chalk.yellowBright(totalSize + " Bytes")
        );

        // Format the size for display
        const sizeInBytes = parseInt(totalSize, 10);
        const size = sizeInBytes
          ? sizeInBytes > 1024
            ? `${(sizeInBytes / 1024).toFixed(2)} KB`
            : `${sizeInBytes} Bytes`
          : null;

        if (size) {
          if (folder.parent_id && folder.parent_id == id) {
            await db.query(
              `UPDATE repository SET size = '${size}' WHERE id = ${folder.id} AND deleted = 0`
            );
          } else {
            await db.query(
              `UPDATE repository SET size = '${size}' WHERE id = ${folder.id} AND deleted = 0`
            );
          }
        }

        // Push folder with updated size to finalData
        finalData.push({
          ...folder,
          size: size,
        });
      } else {
        // Non-folder items can also be included
        // check if file exists
        if (
          folder.url &&
          fs.existsSync(
            path.join(__dirname, `../public${getPathAfterHost(folder.url)}`)
          )
        ) {
          const joinPublicPath = path.join(
            __dirname,
            `../public${getPathAfterHost(folder.url)}`
          );
          const itemStats = folder?.url ? fs.statSync(joinPublicPath) : null;
          const sizeInBytes =
            itemStats.size != null ? parseInt(itemStats.size, 10) : null;
          const size = sizeInBytes
            ? sizeInBytes > 1024
              ? `${(sizeInBytes / 1024).toFixed(2)} KB`
              : `${sizeInBytes} Bytes`
            : null;

          if (size) {
            console.log(
              `Size of file '${folder?.url}': ` +
                chalk.yellowBright(size + " Bytes")
            );
            await db.query(
              "UPDATE repository SET size = ? WHERE id = ? AND deleted = 0",
              [size, folder.id]
            );
          }

          finalData.push({
            ...folder,
            size: size, // Default size for non-folder items in this case
          });
        } else if (getDomain(folder.url) == "storage.googleapis.com") {
          finalData.push(folder);
        }
      }
    }

    return sendResponse(res, 200, finalData);
  } else {
    // Process when folderData contains size key
    const finalData = await Promise.all(
      folderData.map(async (data) => {
        const joinPublicPath = path.join(__dirname, `../public/${data?.url}`);
        // console.log("joinPublicPath: ", joinPublicPath);
        if (data?.url && fs.existsSync(joinPublicPath)) {
          const itemStats = data?.url ? fs.statSync(joinPublicPath) : null;
          const sizeInBytes =
            itemStats.size != null ? parseInt(itemStats.size, 10) : null;
          const size = sizeInBytes
            ? sizeInBytes > 1024
              ? `${(sizeInBytes / 1024).toFixed(2)} KB`
              : `${sizeInBytes} Bytes`
            : null;

          if (size) {
            console.log(
              `Size of file '${data?.url}': ` +
                chalk.yellowBright(size + " Bytes")
            );
            await db.query(
              "UPDATE repository SET size = ? WHERE id = ? AND deleted = 0",
              [size, data.id]
            );
          }

          return {
            ...data,
            size: size,
          };
        } else {
          return data;
        }
      })
    );

    const filteredData = finalData.filter((item) => item !== undefined);

    return sendResponse(res, 200, filteredData);
  }
};

export const getFolderDetails = async (req, res) => {
  let { id } = req.params;

  const condition = await whereCondition({
    table: "repository",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    user: req.user,
    id,
    grouped: req.query.grouped,
  });

  const getFolderData = `SELECT * FROM repository WHERE deleted = 0 ${condition}`;
  const [folderData] = await db.query(getFolderData);
  return sendResponse(res, 200, folderData);
};

export const updateFolder = async (req, res) => {
  const { id, folderName, organization } = req.body;

  if (!id || !folderName || !organization)
    return sendResponse(res, 400, "id,folderName,organization is required");

  const getFolderData = `SELECT * FROM repository WHERE id = ${id} AND deleted = 0 AND organization = ${organization}`;
  const [currentFolder] = await db.query(getFolderData);

  if (!currentFolder[0]) return sendResponse(res, 404, "Folder data not found");

  if (currentFolder[0].is_locked === 1) {
    return sendResponse(
      res,
      400,
      `cannot edit '${currentFolder[0].name}': Permission denied`
    );
  }

  const [parentFolder] = await db.query(
    `SELECT name FROM repository WHERE parent_id = ${currentFolder[0]?.parent_id} AND deleted = 0 AND organization = ${organization} AND type = 'folder' AND id != ${id}`
  );
  const parentFolderNames = parentFolder.map((folder) => folder.name);
  if (parentFolderNames.length && parentFolderNames.includes(folderName))
    return sendResponse(
      res,
      400,
      `This destination already contains a folder named '${folderName}'`
    );
  // console.log(currentFolder[0]?.url);

  const publicPath = path.join(process.cwd(), `public`);
  const folderPath = path.join(publicPath, `${currentFolder[0]?.url}`);
  console.log("folderPath: ", folderPath);
  if (!fs.existsSync(folderPath))
    return sendResponse(res, 404, "Folder not found in the filesystem");

  const newPath = path.join(`${folderPath}/../${folderName}`);
  console.log(chalk.gray(newPath));
  const lastPublicIndex = newPath.lastIndexOf("public");

  let relativeFolderPath = newPath.substring(
    lastPublicIndex + "public\\".length
  );

  try {
    fs.renameSync(folderPath, newPath);
    // console.log(chalk.blue("file renamed successfully"));

    const { query, values } = updateQueryBuilder(Repository, {
      id,
      name: folderName,
      url: relativeFolderPath,
    });
    const [updatedData] = await db.query(query, values);

    await insertActivityLog(
      req.user.sessionid,
      "update",
      "Repository",
      `This user updated Repository with id: ${updatedData?.id}`
    );

    return sendResponse(res, 200, "Folder updated successfully");
  } catch (error) {
    console.error(chalk.red("Error renaming file: "), error);
    return sendResponse(res, 200, "Error renaming file");
  }
};

export const deleteFolder = async (req, res) => {
  const id = req.params?.id;
  const getFolderData = `SELECT * FROM repository WHERE id = ${id} and deleted = 0`;
  const [currentFolder] = await db.query(getFolderData);

  if (!currentFolder.length) return sendResponse(res, 404, `Record not found`);
  if (currentFolder[0]?.is_locked === 1)
    return sendResponse(
      res,
      400,
      `cannot delete '${currentFolder[0]?.name}': Permission denied`
    );

  const publicPath = path.join(process.cwd(), `public`);
  if (currentFolder[0]?.type == "file") {
    const [getFileData] = await db.query(
      `SELECT id,deleting_rights_employees, document_status FROM document_creation WHERE id = ${currentFolder[0]?.document_creation_id} and deleted = 0`
    );
    if (!getFileData.length)
      return sendResponse(res, 404, `${getFileData[0]?.type} not found`);

    const deletingRightsEmployees = JSON.parse(
      getFileData[0]?.deleting_rights_employees || "[]"
    );
    if (
      !deletingRightsEmployees?.includes(req.user.sessionid) &&
      !req.user.isSuperAdmin
    )
      return sendResponse(
        res,
        400,
        `cannot delete '${currentFolder[0]?.name}': Permission denied`
      );

    const { query, values } = updateQueryBuilder(DocumentCreation, {
      id: getFileData[0]?.id,
      deleted: 1,
    });
    await db.query(query, values);

    // await fs.promises.unlink(path.join(publicPath, `${currentFolder[0]?.url}`));

    // Get the URL string from your data
    const fileUrlString = currentFolder[0]?.url;

    if (fileUrlString) {
      // Parse the URL string
      const parsedUrl = new URL(fileUrlString);

      // Extract the pathname (e.g., '/root/1743677827348.jpg')
      // Use decodeURIComponent to handle potential URL-encoded characters in the path
      const relativeFilePath = decodeURIComponent(parsedUrl.pathname);

      // Construct the full local file path
      // path.join handles path separators correctly across different OS
      // We might need to remove a leading '/' from the pathname if publicPath already represents the root
      const filePathToDelete = path.join(
        publicPath,
        relativeFilePath.startsWith("/")
          ? relativeFilePath.substring(1)
          : relativeFilePath
      );

      console.log(`Attempting to delete file at: ${filePathToDelete}`); // For debugging

      // Attempt to delete the file
      await fs.promises.unlink(filePathToDelete);
      console.log(`Successfully deleted file: ${filePathToDelete}`);
    }

    const data = {
      organization: currentFolder[0]?.organization,
      department: currentFolder[0]?.department,
      repository_id: id,
      created_by: req.user.sessionid,
      document_name: currentFolder[0]?.name,
      action_type: "delete",
      document_status: getFileData[0]?.document_status,
    };
    // audit log for document
    await auditLogFunction(data);
  }
  const { query, values } = updateQueryBuilder(Repository, { id, deleted: 1 });
  const [updatedData] = await db.query(query, values);

  if (currentFolder[0].type == "folder") {
    const folderPath = path.join(publicPath, `${currentFolder[0]?.url}`);
    await fs.promises.rm(folderPath, { recursive: true });
  }

  await insertActivityLog(
    req.user.sessionid,
    "delete",
    "Repository",
    `This user deleted Repository with id: ${updatedData?.id}`
  );
  return sendResponse(
    res,
    200,
    `${currentFolder[0]?.type} deleted successfully`
  );
};
