import moment from "moment";
import db from "../db-config.js";
import {
  insertActivityLog,
  storeError,
  getOrganizationAccordingToDepartment,
  whereCondition,
  makeJoins,
  countQueryCondition,
  deleteRecord,
  dateValidator,
  updateQueryBuilder,
  createQueryBuilder,
  searchConditionRecord,
  encodeAndStringifyFields,
  insertNotification,
  decodeAndParseFields,
  getRecord,
  uniqueIdGenerator,
  getDateRangeForSQL,
  generateStrategicTacticalTables,
  sendDynamicEmail,
  riskRegisterTemplateHelper,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import StrategicRiskIdentification from "../sequelize/StrategicRiskIdentificationSchema.js";
import StrategicRisk from "../sequelize/StrategicRiskSchema.js";

export async function sendRiskCreationEmail(req, id) {
  try {
    const { strategicRisk } = await getStrategicRiskAndItsIdentification(
      req,
      id
    );
    const strategicData = strategicRisk[0];
    const companyWebsite = process.env.HARMONY_FRONTEND_URL;

    const {
      controlIdentificationTable,
      inherentRiskTable,
      opportunityIdentificationTable,
      overallResidualRatingTable,
      residualRatingTable,
      strategicRiskIdentifiedTable,
    } = generateStrategicTacticalTables(strategicData);

    let usersToSend = [
      {
        name: `${strategicData.facilitator_name} ${strategicData.facilitator_surname}`,
        email: strategicData.facilitator_email,
      },
      {
        name: `${strategicData.approver_name} ${strategicData.approver_surname}`,
        email: strategicData.approver_email,
      },
    ];
    strategicData.risk_identification_details.forEach((risk) => {
      usersToSend.push({
        name: `${risk.risk_owner_name} ${risk.risk_owner_surname}`,
        email: risk.risk_owner_email,
      });
    });

    for (let user of usersToSend) {
      const sendRecordArray = {
        name: user.name,
        templateFileUrl: "mail_for_strategic_tactical_register_template.html",
        templateName: "Strategic Risk/ Tactical Risk Register creation",
        company_name: strategicData.organization_name,
        company_website: companyWebsite,
        risk_register_name: strategicData.name,
        risk_assessment_date: moment(strategicData.assessment_date).format(
          "DD-MM-YYYY"
        ),
        status: strategicData.status,
        business_structure: strategicData.organization_name,
        risk_review_date: moment(strategicData.review_date).format(
          "DD-MM-YYYY"
        ),
        risk_assessment_facilitator: `${strategicData.facilitator_name} ${strategicData.facilitator_surname}`,
        risk_register_approver: `${strategicData.approver_name} ${strategicData.approver_surname}`,
        risk_meeting_name: strategicData.meeting,
        strategicRisksIdentified: strategicRiskIdentifiedTable,
        inherentRisk: inherentRiskTable,
        controlIdentification: controlIdentificationTable,
        residualRating: residualRatingTable,
        overallResidualRating: overallResidualRatingTable,
        opportunityIdentification: opportunityIdentificationTable,
      };
      await sendDynamicEmail({
        to: user.email,
        subject: `Successful Creation of Risk Register - ${strategicData.name}`,
        data: sendRecordArray,
      });
    }
    return true;
  } catch (error) {
    console.log("Error in sending risk creation email: ", error);
  }
}

/**Function to create/update strategic risk   */
export const strategicRiskCreateUpdate = async (req, res) => {
  const {
    id,
    risk_register_name,
    assessment_date,
    review_date,
    risk_context,
    risk_assessment_facilitator,
    risk_register_approver,
    meeting_id,
    status,
    organization,
    department,
  } = req.body;
  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization =
      await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  req.body.assessment_date = dateValidator(assessment_date);
  req.body.review_date = dateValidator(review_date);
  let strategicRiskIdentificationList = req.body.risk_identification;
  /**If id comes in body then it will update the query */
  if (id) {
    req.body.updated_by = req.user.sessionid;
    /** add a check whether strategic risk is rejected before updating */
    const [statusCheck] = await getRecord("strategic_risk", "id", id);
    if (statusCheck.approval_status == "Rejected") {
      return sendResponse(res, 400, "Record already rejected !");
    }
    /**Update StrategicRisk Query */
    const { query, values } = updateQueryBuilder(StrategicRisk, req.body);
    await db.query(query, values);

    if (status === "Complete" && statusCheck.status == "Draft") {
      const templateFileUrl =
        "mail_for_strategic_tactical_register_approval_template.html";
      const templateName = "Strategic Risk/ Tactical Risk Register Approval";
      const subject = `Risk Register Awaiting Your Approval - ${risk_register_name}`;
      await riskRegisterTemplateHelper(
        templateFileUrl,
        templateName,
        subject,
        id
      );
    }

    if (
      statusCheck.approval_status == "Pending Approval" &&
      req.body.approval_status == "Approved"
    ) {
      const templateFileUrl =
        "mail_for_strategic_tactical_register_approved_template.html";
      const templateName = "Strategic Risk/ Tactical Risk Register Approved";
      const subject = `Risk Register Approval - ${risk_register_name}`;
      await riskRegisterTemplateHelper(
        templateFileUrl,
        templateName,
        subject,
        id
      );
    }

    /** also update the all risk identification related to this strategic risk */
    for (let i = 0; i < strategicRiskIdentificationList.length; i++) {
      let riskIdentification = strategicRiskIdentificationList[i];
      riskIdentification.risk_register_id = id;
      riskIdentification.updated_by = req.user.sessionid;
      riskIdentification.organization = req.body.organization;
      riskIdentification.department = req.body.department;
      const riskIdentificationId = riskIdentification.id;
      riskIdentification = await encodeAndStringifyFields(riskIdentification);
      const { query, values } = riskIdentificationId
        ? updateQueryBuilder(StrategicRiskIdentification, riskIdentification)
        : createQueryBuilder(StrategicRiskIdentification, riskIdentification);
      await db.query(query, values);
    }
    /**Insert Activity  */
    await insertActivityLog(req.user.sessionid, "update", "StrategicRisk", id);
    const [record] = await getRecord("strategic_risk", "id", id);
    // console.log("record: ", record);
    /// send a notification to risk owner that strategic risk has been created
    if (status == "Complete" && record.approval_status != "Rejected") {
      // console.log("record.approval_status: ", record.approval_status);
      const userId = req.body?.risk_register_approver;
      const pageUrl = `/risk/strategic-risk-register?tab=approval-workflow`;
      if (userId) {
        await insertNotification(
          "Strategic Risk",
          pageUrl,
          userId,
          "url",
          req.user.sessionid
        );
      }
    }

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    const unique_id = await uniqueIdGenerator(
      organizationId,
      department,
      "STR",
      "strategic_risk",
      "unique_id",
      "unique_id"
    );
    const type = "reference";
    const reference_no = await uniqueIdGenerator(
      organizationId,
      department,
      "STR",
      "strategic_risk",
      "reference_no",
      type
    );
    req.body.unique_id = unique_id;
    req.body.reference_no = reference_no;
    req.body.created_by = req.user.sessionid;
    /**Insert record for StrategicRisk */
    const { query, values } = createQueryBuilder(StrategicRisk, req.body);
    const [createStrategicRisk] = await db.query(query, values);

    /** also add the all risk_identification related to this strategic risk */
    for (let i = 0; i < strategicRiskIdentificationList.length; i++) {
      let riskIdentification = strategicRiskIdentificationList[i];
      riskIdentification.risk_register_id = createStrategicRisk.insertId;
      riskIdentification.updated_by = req.user.sessionid;
      riskIdentification.organization = req.body.organization;
      riskIdentification.department = req.body.department;
      riskIdentification = await encodeAndStringifyFields(riskIdentification);
      const { query, values } = createQueryBuilder(
        StrategicRiskIdentification,
        riskIdentification
      );
      await db.query(query, values);
    }

    await sendRiskCreationEmail(req, createStrategicRisk.insertId);

    /**Insert record for activity log */
    await insertActivityLog(
      req.user.sessionid,
      "create",
      "Strategic Risk",
      createStrategicRisk.insertId
    );
    const [record] = await getRecord(
      "strategic_risk",
      "id",
      createStrategicRisk.insertId
    );
    /// send a notification to risk owner that strategic risk has been created
    if (status == "Complete" && record.approval_status != "Rejected") {
      const userId = req.body?.risk_register_approver;
      const pageUrl = `/risk/strategic-risk-register?tab=approval-workflow`;
      if (userId) {
        await insertNotification(
          "Strategic Risk",
          pageUrl,
          userId,
          "url",
          req.user.sessionid
        );
      }
    }

    return sendResponse(res, 200, "Record created successfully");
  }
};

export async function getStrategicRiskAndItsIdentification(req, id) {
  const condition = await whereCondition({
    table: "strategic_risk",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id,
    grouped: req.query.grouped,
    user: req.user,
  });

  const searchTableName = [
    "CONCAT(facilitator.name , ' ' , facilitator.surname)",
    "CONCAT(users.name , ' ' , users.surname)",
    "CONCAT(approver.name , ' ' , approver.surname)",
    "strategic_risk.risk_register_name",
    "strategic_risk.risk_context",
    "strategic_risk.status",
    "strategic_risk.approval_status",
    "strategic_risk.unique_id",
    "strategic_risk.reference_no",
    "meeting.meeting_title",
    "organization.name",
  ];
  /** If value come with any search condition then search that word */
  let searchCondition = await searchConditionRecord(
    req.query.search,
    searchTableName
  );

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = strategic_risk.created_by",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = strategic_risk.organization",
    },
    {
      type: "left",
      targetTable: "meeting",
      onCondition: "meeting.id = strategic_risk.meeting_id",
    },
    {
      type: "left",
      targetTable: "users as facilitator",
      onCondition:
        "facilitator.id = strategic_risk.risk_assessment_facilitator",
    },
    {
      type: "left",
      targetTable: "users as approver",
      onCondition: "approver.id = strategic_risk.risk_register_approver",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all strategicRisk */
  const strategicRiskQuery = `SELECT
  strategic_risk.reject_comment, 
  strategic_risk.approval_status,
  strategic_risk.id,
  strategic_risk.unique_id,
   strategic_risk.reference_no,
   strategic_risk.created_by as created_by_id,
   strategic_risk.organization,
   organization.name as organization_name,
   users.name as created_by,
   users.surname as created_by_surname , users.profile as created_by_profile,strategic_risk.organization,strategic_risk.risk_register_name, strategic_risk.assessment_date , strategic_risk.review_date , strategic_risk.risk_context , strategic_risk.risk_assessment_facilitator , strategic_risk.register_assessment_facilitator , strategic_risk.risk_register_approver , strategic_risk.meeting_id ,strategic_risk.status, meeting.meeting_title as meeting , facilitator.name as facilitator_name , facilitator.surname as facilitator_surname , facilitator.profile as facilitator_profile ,facilitator.email as facilitator_email, approver.name as approver_name,approver.surname as approver_surname ,approver.profile as approver_profile, approver.email as approver_email 
      FROM strategic_risk 
      ${joinsRecord} where strategic_risk.deleted = 0  ${searchCondition} ${condition}`;
  let [strategicRisk] = await db.query(strategicRiskQuery);

  strategicRisk = await decodeAndParseFields(strategicRisk);

  for (let strategy of strategicRisk) {
    const id = strategy?.id;
    let [dataFetch] = await db.query(
      `SELECT strategic_risk_identification.* , users.name as created_by,users.surname as created_by_surname,users.profile as created_by_profile , strategic_risk .risk_register_name  , control.name as risk_control_owner_name,control.surname as risk_control_owner_surname, control.profile as risk_control_owner_profile , riskOwner.name as risk_owner_name,riskOwner.surname as risk_owner_surname , riskOwner.profile as risk_owner_profile , riskOwner.email as risk_owner_email, organization.name as organization_name FROM strategic_risk_identification LEFT JOIN users ON users.id = strategic_risk_identification.created_by LEFT JOIN users as control ON control.id = strategic_risk_identification.id LEFT JOIN users as riskOwner ON riskOwner.id = strategic_risk_identification.risk_owner LEFT JOIN organization ON organization.id = strategic_risk_identification.organization LEFT JOIN strategic_risk ON strategic_risk.id = strategic_risk_identification.risk_register_id WHERE risk_register_id = '${id}' AND strategic_risk_identification.deleted = 0`
    );
    dataFetch = await decodeAndParseFields(dataFetch);
    strategy.risk_identification_details = dataFetch;
  }
  return { strategicRisk, strategicRiskQuery };
}

/**Function to view all and single Strategic Risk */
export const viewStrategicRisk = async (req, res) => {
  const { id } = req.params;
  const { strategicRisk, strategicRiskQuery } =
    await getStrategicRiskAndItsIdentification(req, id);
  const totalRecord = await countQueryCondition(strategicRiskQuery);

  return sendResponse(res, 200, strategicRisk, totalRecord);
};

/**Function to delete a specific Strategic Risk */
export const deleteStrategicRisk = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("strategic_risk", id);
  // also deleted identification
  await deleteRecord("strategic_risk_identification", id, "risk_register_id");
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    insertActivityLog(req.user.sessionid, "delete", "Strategic Risk", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
  return sendResponse(res, 404, "Record not found");
};
