import db from "../db-config.js";
import {
  checkNameInOrganization,
  countQueryCondition,
  createQueryBuilder,
  decodeSingle_statement,
  deleteRecord,
  encodeSingle_statement,
  getOrganizationAccordingToDepartment,
  insertActivityLog,
  makeJoins,
  makeLoopAndGetMultipleUsersDetails,
  searchConditionRecord,
  uniqueIdGenerator,
  updateQueryBuilder,
  whereCondition,
} from "../helper/general.js";
import { sendResponse } from "../helper/wrapper.js";
import Team from "../sequelize/TeamSchema.js";

/**Function to create new teams for employee  */
export const createUpdateTeams = async (req, res) => {
  const { id, name, organization, description, department } = req.body;
  req.body.description = await encodeSingle_statement(description);

  /**Check record if organization is not coming then fetch organization according to department */
  let organizationId = organization;
  if (department) {
    const recordAccordingToOrganization = await getOrganizationAccordingToDepartment(department);
    organizationId = recordAccordingToOrganization[0].organization;
    req.body.organization = organizationId;
  }

  /**Check that this organization have that particular name or not*/
  const checkNameWithOrganization = await checkNameInOrganization("teams", name, organizationId);

  /**If id comes in body then it will update the query */
  // req.body.team_members = JSON.stringify(team_members);
  if (id) {
    req.body.updated_by = req.user.sessionid;
    if (checkNameWithOrganization.length > 0 && checkNameWithOrganization[0].id != id) {
      return sendResponse(res, 400, `Record with name ${name} already exists`);
    }
    /**Update Teams Query */
    const { query, values } = updateQueryBuilder(Team, req.body);

    await db.query(query, values);

    /**Insert Activity  */
    await insertActivityLog(req.user.sessionid, "update", "Teams", id);

    return sendResponse(res, 200, "Record updated successfully");
  } else {
    if (checkNameWithOrganization.length > 0) {
      return sendResponse(res, 400, "This record already exists");
    }
    req.body.created_by = req.user.sessionid;

    const unique_id = await uniqueIdGenerator(organization, department, "Team", "teams", "unique_id", "unique_id");
    req.body.unique_id = unique_id;

    /**Insert record for teams */
    const { query, values } = createQueryBuilder(Team, req.body);

    const [createTeams] = await db.query(query, values);

    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "create", "teams", createTeams.insertId);

    return sendResponse(res, 200, "Record created successfully");
  }
};

/**Function to view all and single team */
export const viewAllTeams = async (req, res) => {
  const { id } = req.params;

  const condition = await whereCondition({
    table: "teams",
    page: req.query.page,
    all: req.query.all,
    pageSize: req.query.pageSize,
    filter: req.query.filter,
    id: id,
    user: req.user,
    grouped: req.query.grouped,
  });

  const searchTableName = [
    "teams.name",
    "teams.description",
    "CONCAT(teamLeader.name , ' ' , teamLeader.surname)",
    "CONCAT(users.name , ' ' , users.surname)",
    "organization.name",
    "department.name",
  ];
  let searchCondition = await searchConditionRecord(req.query.search, searchTableName);

  /**Make Joins according to tables */
  const joins = [
    {
      type: "left",
      targetTable: "users",
      onCondition: "users.id = teams.created_by",
    },
    {
      type: "left",
      targetTable: "users as teamLeader",
      onCondition: "teamLeader.id = teams.team_leader",
    },
    {
      type: "left",
      targetTable: "department",
      onCondition: "department.id = teams.department",
    },
    {
      type: "left",
      targetTable: "organization",
      onCondition: "organization.id = teams.organization",
    },
  ];

  const joinsRecord = await makeJoins(joins);

  /**Record of all teams */
  const teamsQuery = `SELECT teams.id,teams.unique_id, teams.created_by as created_by_id,teams.name,teams.description,teams.department, department.name as department_name, teams.team_leader,organization.name as organization_name,teams.organization as organization ,users.name as created_by , users.surname as created_by_surname , users.profile as created_by_profile,teamLeader.name as team_leader_name , teamLeader.surname as team_leader_surname , teamLeader.profile as team_leader_profile
    FROM teams ${joinsRecord} where teams.deleted = 0 ${searchCondition} ${condition}`;
  const [teams] = await db.query(teamsQuery);
  // teams.forEach(async (team) => {
  //   team.description = await decodeSingle_statement(team.description);
  // });

  for (const team of teams) {
    team.description = await decodeSingle_statement(team.description);
  }

  /**Make loop for getting record of teams members details according to team leaders */
  // if (teams.length > 0) {
  //   await makeLoopAndGetMultipleUsersDetails(teams, "members");
  // }

  const result = teams;

  /**Count of all teams */
  const totalRecord = await countQueryCondition(teamsQuery);

  return sendResponse(res, 200, result, totalRecord);
};

/**Function to delete a specific team */
export const deleteTeams = async (req, res) => {
  const { id } = req.params;
  const deleteRecordQuery = await deleteRecord("teams", id);
  if (deleteRecordQuery) {
    /**Insert record for activity log */
    await insertActivityLog(req.user.sessionid, "delete", "Teams", id);
    return sendResponse(res, 200, "Record deleted successfully");
  }
};
