// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateAccessorMethod,
  deleteAccessorMethod,
  viewAccessorMethod,
} from "../controller/assessorMethodController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Assessor Method
 *   description: API operations for managing all Assessor Method
 */
const assessorMethodRoutes = Router();

/**
 * @swagger
 * /api/assessorMethod/create-update-assessor-method:
 *   post:
 *     summary: Create or update an Assessor Method
 *     tags: [Assessor Method]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Assessor Method"
 *             description: "Description of the accessor method"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Assessor Method created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
assessorMethodRoutes.post(
  "/create-update-assessor-method",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      317,
      "assessor_method"
    )(req, res, next);
  },
  controllerWrapper(createUpdateAccessorMethod)
);

/**
 * @swagger
 * /api/assessorMethod/get-assessor-method:
 *   get:
 *     summary: Get list of all Assessor Methods
 *     tags: [Assessor Method]
 *     responses:
 *       200:
 *         description: List of accessor methods
 *       500:
 *         description: Internal server error
 */
assessorMethodRoutes.get(
  "/get-assessor-method",
  verifyToken,
  sideBarPermissionCheck("view", 317, "assessor_method"),
  controllerWrapper(viewAccessorMethod)
);

/**
 * @swagger
 * /api/assessorMethod/get-assessor-method/{id}:
 *   get:
 *     summary: Get a specific record of Assessor Method
 *     tags: [Assessor Method]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the accessor method
 *     responses:
 *       200:
 *         description: Assessor Method data
 *       404:
 *         description: Assessor Method not found
 *       500:
 *         description: Internal server error
 */
assessorMethodRoutes.get(
  "/get-assessor-method/:id",
  verifyToken,
  sideBarPermissionCheck("view", 317, "assessor_method"),
  controllerWrapper(viewAccessorMethod)
);

/**
 * @swagger
 * /api/assessorMethod/delete-assessor-method/{id}:
 *   delete:
 *     summary: Delete a specific record of Assessor Method
 *     tags: [Assessor Method]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the accessor method to delete
 *     responses:
 *       200:
 *         description: Assessor Method successfully deleted
 *       404:
 *         description: Assessor Method not found
 *       500:
 *         description: Internal server error
 */
assessorMethodRoutes.delete(
  "/delete-assessor-method/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 317, "assessor_method"),
  controllerWrapper(deleteAccessorMethod)
);

export default assessorMethodRoutes;
