// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedAuditCategory,
  deleteAuditCategory,
  viewAllAuditCategory,
} from "../controller/auditCategoryController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { auditCategoryCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Audit Category
 *   description: API operations for managing all Audit Categories
 */
const auditCategoryRoutes = Router();

/**
 * @swagger
 * /api/auditCategory/create-update-audit-category:
 *   post:
 *     summary: Create or update an Audit Category
 *     tags: [Audit Category]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example: { id: "id", name: "name", description: "description", organization: [] }
 */
auditCategoryRoutes.post(
  "/create-update-audit-category",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      143,
      "audit_category"
    )(req, res, next);
  },
  validateRequest(auditCategoryCreateUpdateValidationSchema),
  controllerWrapper(createUpdatedAuditCategory)
);

/**
 * @swagger
 * /api/auditCategory/get-audit-category:
 *   get:
 *     summary: Get a list of all Audit Categories
 *     tags: [Audit Category]
 */
auditCategoryRoutes.get(
  "/get-audit-category",
  verifyToken,
  sideBarPermissionCheck("view", 143, "audit_category"),
  controllerWrapper(viewAllAuditCategory)
);

/**
 * @swagger
 * /api/auditCategory/get-audit-category/{id}:
 *   get:
 *     summary: Get a specific record of Audit Category
 *     tags: [Audit Category]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit category to retrieve
 *         schema:
 *           type: string
 */
auditCategoryRoutes.get(
  "/get-audit-category/:id",
  verifyToken,
  sideBarPermissionCheck("view", 143, "audit_category"),
  controllerWrapper(viewAllAuditCategory)
);

/**
 * @swagger
 * /api/auditCategory/delete-audit-category/{id}:
 *   delete:
 *     summary: Delete a specific record of Audit Category
 *     tags: [Audit Category]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit category to delete
 *         schema:
 *           type: string
 */
auditCategoryRoutes.delete(
  "/delete-audit-category/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 143, "audit_category"),
  controllerWrapper(deleteAuditCategory)
);

export default auditCategoryRoutes;
