// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  addEvaluationTool,
  createUpdateAuditScheduling,
  deleteAuditScheduling,
  getSpecificRecords,
  uploadReport,
  viewAllAuditScheduling,
} from "../controller/auditSchedulingController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { auditSchedulingCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";
import { saveAuditRecord } from "../controller/auditExecutionController.js";
/**
 * @swagger
 * tags:
 *   name: Audit Scheduling
 *   description: API operations for managing Audit Scheduling
 */
const auditSchedulingRoutes = Router();

/**
 * @swagger
 * /api/auditScheduling/create-update-audit-scheduling:
 *   post:
 *     summary: Create or update an Audit Scheduling
 *     tags: [Audit Scheduling]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: integer
 *               name:
 *                 type: string
 *               type:
 *                 type: integer
 *               linked_to:
 *                 type: string
 *               module:
 *                 type: integer
 *               sub_module:
 *                 type: integer
 *               record_name:
 *                 type: integer
 *               planned_start_date:
 *                 type: string
 *                 format: date
 *               planned_end_date:
 *                 type: string
 *                 format: date
 *               want_to_repeat:
 *                 type: integer
 *                 nullable: true
 *               when:
 *                 type: string
 *               cycles:
 *                 type: string
 *               source:
 *                 type: integer
 *               lead_person:
 *                 type: integer
 *               members:
 *                 type: string
 *               owner:
 *                 type: string
 *               participants:
 *                 type: string
 *               scope:
 *                 type: string
 *               critical_business:
 *                 type: string
 *               financial_resource:
 *                 type: string
 *               value:
 *                 type: integer
 *               currency:
 *                 type: integer
 *               opening_meeting_agenda:
 *                 type: string
 *                 format: date
 *               closing_meeting_agenda:
 *                 type: string
 *                 format: date
 *               evaluation_tool:
 *                 type: string
 *               organization:
 *                 type: integer
 *                 nullable: true
 *               created_at:
 *                 type: string
 *                 format: date-time
 *               created_by:
 *                 type: integer
 *               deleted:
 *                 type: integer
 *               updated_at:
 *                 type: string
 *                 format: date-time
 *               updated_by:
 *                 type: integer
 *               department:
 *                 type: integer
 *           example: {
 *             id: 1,
 *             name: "Audit Name",
 *             schedule_type: "Audit , Inspection , Surveillance",
 *             type: 1,
 *             reference_number : 1,
 *             linked_to: "Linked Entity",
 *             module: 1,
 *             sub_module: 2,
 *             record_name: 101,
 *             planned_start_date: "2024-01-01",
 *             planned_end_date: "2024-01-02",
 *             want_to_repeat: 0,
 *             when: "Monthly",
 *             cycles: "12",
 *             source: 3,
 *             lead_person: 4,
 *             members: "Auditor Member List",
 *             owner: "Audit Owner",
 *             participants: "Participants List",
 *             scope: "Audit Scope",
 *             critical_business: "Critical Business Area",
 *             financial_resource: "Financial Resources",
 *             value: 10000,
 *             currency: 1,
 *             opening_meeting_agenda: "2024-01-01",
 *             closing_meeting_agenda: "2024-01-02",
 *             evaluation_tool: "Tool Name",
 *             organization: 1,
 *             created_at: "2024-01-01T00:00:00Z",
 *             created_by: 1,
 *             deleted: 0,
 *             updated_at: "2024-01-01T00:00:00Z",
 *             updated_by: 2,
 *             department: 3
 *           }
 */
auditSchedulingRoutes.post(
  "/create-update-audit-scheduling",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 155, "audit_scheduling")(req, res, next);
  },
  // validateRequest(auditSchedulingCreateUpdateValidationSchema),
  controllerWrapper(createUpdateAuditScheduling)
);

/**
 * @swagger
 * /api/auditScheduling/get-audit-scheduling:
 *   get:
 *     summary: Get a list of all Audit Scheduling records
 *     tags: [Audit Scheduling]
 */
auditSchedulingRoutes.get(
  "/get-audit-scheduling",
  verifyToken,
  sideBarPermissionCheck("view", 155, "audit_scheduling"),
  controllerWrapper(viewAllAuditScheduling)
);

/**
 * @swagger
 * /api/auditScheduling/get-audit-scheduling/{id}:
 *   get:
 *     summary: Get a specific record of Audit Scheduling
 *     tags: [Audit Scheduling]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit scheduling to retrieve
 *         schema:
 *           type: integer
 */
auditSchedulingRoutes.get(
  "/get-audit-scheduling/:id",
  verifyToken,
  sideBarPermissionCheck("view", 155, "audit_scheduling"),
  controllerWrapper(viewAllAuditScheduling)
);

/**
 * @swagger
 * /api/auditScheduling/get-specific-records/{tableName}:
 *   get:
 *     summary: Get a list of all Specific records
 *     tags: [Audit Scheduling]
 */
auditSchedulingRoutes.get("/get-specific-records", verifyToken, controllerWrapper(getSpecificRecords));

/**
 * @swagger
 * /api/auditScheduling/delete-audit-scheduling/{id}:
 *   delete:
 *     summary: Delete a specific record of Audit Scheduling
 *     tags: [Audit Scheduling]
 *     parameters:
 *       - name: id
 *         in: path
 *         required: true
 *         description: ID of the audit scheduling to delete
 *         schema:
 *           type: integer
 */
auditSchedulingRoutes.delete(
  "/delete-audit-scheduling/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 155, "audit_scheduling"),
  controllerWrapper(deleteAuditScheduling)
);

export default auditSchedulingRoutes;

/**
 * @swagger
 * /api/auditScheduling/save-audit-record:
 *   post:
 *     summary: Save a specific audit record using audit template
 *     tags: [Audit Execution]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *               department:
 *                 type: string
 *               questions:
 *                 type: array
 *                 items:
 *                   type: string
 *               audit_execution:
 *                 type: string
 *           example: { name: "name", description: "Description", organization: [], department: "department", questions: [], audit_execution: "audit_execution" }
 */
auditSchedulingRoutes.post(
  "/save-audit-record",
  verifyToken,
  sideBarPermissionCheck("view", 155, "audit_scheduling"),
  // validateRequest(auditTemplateCreateUpdateValidationSchema),
  controllerWrapper(saveAuditRecord)
);

/**
 * @swagger
 * /api/auditScheduling/upload-report:
 *   post:
 *     summary: upload a report for Audit Scheduling
 *     tags: [Audit Scheduling]
 */
auditSchedulingRoutes.post("/upload-report", verifyToken, controllerWrapper(uploadReport));

/**
 * @swagger
 * /api/auditScheduling/add-evaluation-tool:
 *   post:
 *     summary: Add Evaluation Tool
 *     tags: [Audit Scheduling]
 */
auditSchedulingRoutes.post("/add-evaluation-tool", verifyToken, controllerWrapper(addEvaluationTool));
