import { Router } from "express";
import { controllerWrapper } from "../helper/wrapper.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  bcpComment,
  createUpdateBcpManagement,
  deleteBcpManagement,
  getBcpManagement,
  getBcpManagementApprovalWorkflow,
} from "../controller/bcpManagementController.js";

const BcpManagementRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: BCP Management
 *   description: API endpoints for managing BCP Management
 */

/**
 * @swagger
 * /api/bcpManagement/create-update-bcp-management:
 *   post:
 *     summary: Create or update a BCP Management entry
 *     tags:
 *       - BCP Management
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               responsible_person:
 *                 type: string
 *               tasks:
 *                 type: string
 *               timeline:
 *                 type: string
 *                 format: date
 *               status:
 *                 type: string
 *           example:
 *             name: Emergency Plan Development
 *             description: Development of the company's emergency response plan
 *             responsible_person: Jane Doe
 *             tasks: "Plan writing, Resource allocation"
 *             timeline: "2024-12-15"
 *             status: "In Progress"
 *     responses:
 *       '200':
 *         description: BCP Management entry created or updated successfully
 *       '401':
 *         description: Unauthorized, invalid token
 *       '500':
 *         description: Server error
 */
BcpManagementRoutes.post(
  "/create-update-bcp-management",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      434,
      "bcp_management"
    )(req, res, next);
  },
  controllerWrapper(createUpdateBcpManagement)
);

/**
 * @swagger
 * /api/bcpManagement/get-bcp-management:
 *   get:
 *     summary: Get all BCP Management entries
 *     tags:
 *       - BCP Management
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       '200':
 *         description: Successfully retrieved BCP Management entries
 *       '401':
 *         description: Unauthorized, invalid token
 *       '500':
 *         description: Server error
 */
BcpManagementRoutes.get(
  "/get-bcp-management",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck("view", 434, "bcp_management")(req, res, next);
  },
  controllerWrapper(getBcpManagement)
);

/**
 * @swagger
 * /api/bcpManagement/get-bcp-management/{id}:
 *   get:
 *     summary: Get a BCP Management entry by ID
 *     tags:
 *       - BCP Management
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the BCP Management entry to retrieve
 *     responses:
 *       '200':
 *         description: Successfully retrieved the BCP Management entry
 *       '401':
 *         description: Unauthorized, invalid token
 *       '404':
 *         description: BCP Management entry not found
 *       '500':
 *         description: Server error
 */
BcpManagementRoutes.get(
  "/get-bcp-management/:id",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck("view", 434, "bcp_management")(req, res, next);
  },
  controllerWrapper(getBcpManagement)
);

/**
 * @swagger
 * /api/bcpManagement/delete-bcp-management/{id}:
 *   delete:
 *     summary: Delete a BCP Management entry by ID
 *     tags:
 *       - BCP Management
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: ID of the BCP Management entry to delete
 *     responses:
 *       '200':
 *         description: BCP Management entry deleted successfully
 *       '401':
 *         description: Unauthorized, invalid token
 *       '404':
 *         description: BCP Management entry not found
 *       '500':
 *         description: Server error
 */
BcpManagementRoutes.delete(
  "/delete-bcp-management/:id",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck("delete", 434, "bcp_management")(req, res, next);
  },
  controllerWrapper(deleteBcpManagement)
);



/**
 * @swagger
 * /api/bcpManagement/get-bcp-management-approval-workflow:
 *   get:
 *     summary: Get BCP Management approval workflow
 *     tags:
 *       - BCP Management
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: header
 *         name: Authorization
 *         required: true
 *         schema:
 *           type: string
 *         description: Bearer token for authorization
 *     responses:
 *       '200':
 *         description: Successfully retrieved BCP Management approval workflow
 *       '401':
 *         description: Unauthorized, invalid token
 *       '403':
 *         description: Forbidden, insufficient permissions
 *       '500':
 *         description: Server error
 */
BcpManagementRoutes.get(
  "/get-bcp-management-approval-workflow",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck("view", 434, "bcp_management")(req, res, next);
  },
  controllerWrapper(getBcpManagementApprovalWorkflow)
);
 


/**
 * @swagger
 * /api/bcpManagement/bcp-comment:
 *   post:
 *     summary: Add a comment to a BCP Management entry
 *     tags:
 *       - BCP Management
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               status:
 *                 type: string
 *                 enum: [approved, rejected]
 *                 description: Status of the BCP management entry
 *               bcp_management_id:
 *                 type: integer
 *                 description: ID of the BCP management entry
 *               approver:
 *                 type: integer
 *                 description: ID of the approver
 *               comments:
 *                 type: string
 *                 description: Comments about the BCP management entry
 *             required:
 *               - status
 *               - bcp_management_id
 *     responses:
 *       '200':
 *         description: Comment added successfully
 *       '400':
 *         description: Invalid input data
 *       '401':
 *         description: Unauthorized, invalid token
 *       '403':
 *         description: Forbidden, insufficient permissions
 *       '500':
 *         description: Server error
 */
BcpManagementRoutes.post(
  "/bcp-comment",
  verifyToken,
  sideBarPermissionCheck("view", 69, "bcp_management"),
  controllerWrapper(bcpComment)
);
export default BcpManagementRoutes;
