import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateBestPractice,
  deleteBestPractice,
  viewBestPractice,
} from "../controller/bestPracticeController.js";

/**
 * @swagger
 * tags:
 *   name:  best practice
 *   description: API operations for managing  best practice
 */
const bestPracticeRoutes = Router();

/**
 * @swagger
 * /api/bestPractice/create-update-best-practice:
 *   post:
 *     summary: Create or update a  best practice
 *     tags: [ best practice]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { id: "id", title: "title", description: "description" }
 */
bestPracticeRoutes.post(
  "/create-update-best-practice",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      391,
      "best_practice"
    )(req, res, next);
  },
  controllerWrapper(createUpdateBestPractice)
);

/**
 * @swagger
 * /api/bestPractice/get-best-practice:
 *   get:
 *     summary: Get all records of  best practice
 *     tags: [ best practice]
 */
bestPracticeRoutes.get(
  "/get-best-practice",
  verifyToken,
  sideBarPermissionCheck("view", 391, "best_practice"),
  controllerWrapper(viewBestPractice)
);

/**
 * @swagger
 * /api/bestPractice/get-best-practice/{id}:
 *   get:
 *     summary: Get a specific record of  best practice
 *     tags: [ best practice]
 */
bestPracticeRoutes.get(
  "/get-best-practice/:id",
  verifyToken,
  sideBarPermissionCheck("view", 391, "best_practice"),
  controllerWrapper(viewBestPractice)
);

/**
 * @swagger
 * /api/bestPractice/delete-best-practice/{id}:
 *   delete:
 *     summary: Delete a specific record of  best practice
 *     tags: [ best practice]
 */
bestPracticeRoutes.delete(
  "/delete-best-practice/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 391, "best_practice"),
  controllerWrapper(deleteBestPractice)
);

export default bestPracticeRoutes;
