import { Router } from "express";
import {
  createUpdateContractorMeeting,
  viewAllMeeting,
  deleteContractorMeeting,
} from "../controller/contractorMeetingController.js";

import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { verifyToken } from "../helper/tokenVerify.js";

/**
 * @swagger
 * tags:
 *   name: Contractor Meeting
 *   description: API operations for managing contractor meetings
 */

const contractorMeetingRoutes = Router();

/**
 * @swagger
 * /api/contractorMeeting/create-contractor-meeting:
 *   post:
 *     summary: Create or update a contractor meeting
 *     tags: [Contractor Meeting]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               meeting_id:
 *                 type: string
 *                 example: "12345"
 *               meeting_title:
 *                 type: string
 *                 example: "Quarterly Safety Review"
 *               meeting_hierarchy:
 *                 type: string
 *                 example: "Level 1"
 *               participants:
 *                 type: string
 *                 example: "John Doe, Jane Smith"
 *               meeting_owner:
 *                 type: integer
 *                 example: 3
 *               planned_meeting_date_from:
 *                 type: string
 *                 format: date-time
 *                 example: "2024-09-01T09:00:00Z"
 *               planned_meeting_date_to:
 *                 type: string
 *                 format: date-time
 *                 example: "2024-09-01T10:00:00Z"
 *               venue:
 *                 type: string
 *                 example: "Conference Room A"
 *               agenda:
 *                 type: string
 *                 example: "Discuss safety protocols and improvements."
 *               organization:
 *                 type: integer
 *                 example: 5
 *               department:
 *                 type: integer
 *                 example: 2
 *               created_by:
 *                 type: integer
 *                 example: 1
 *               updated_by:
 *                 type: integer
 *                 example: 2
 *               deleted:
 *                 type: integer
 *                 example: 0
 *               created_at:
 *                 type: string
 *                 format: date-time
 *                 example: "2024-09-01T09:00:00Z"
 *               updated_at:
 *                 type: string
 *                 format: date-time
 *                 example: "2024-09-01T09:30:00Z"
 *     responses:
 *       200:
 *         description: Contractor Meeting created or updated successfully.
 *       400:
 *         description: Bad Request. Invalid input data.
 *       401:
 *         description: Unauthorized. Authentication is required.
 *       500:
 *         description: Internal Server Error.
 */

contractorMeetingRoutes.post(
  "/create-contractor-meeting",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      271,
      "contractor_meeting"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorMeeting)
);

/**
 * @swagger
 * /api/contractorMeeting/get-contractor-meeting:
 *   get:
 *     summary: Get all contractor meetings
 *     tags: [Contractor Meeting]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: A list of contractor meetings.
 */

contractorMeetingRoutes.get(
  "/get-contractor-meeting",
  verifyToken,
  sideBarPermissionCheck("view", 271, "contractor_meeting"),
  controllerWrapper(viewAllMeeting)
);

/**
 * @swagger
 * /api/contractorMeeting/get-contractor-meeting/{id}:
 *   get:
 *     summary: Get a specific contractor meeting by ID
 *     tags: [Contractor Meeting]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor meeting
 *     responses:
 *       200:
 *         description: Details of the contractor meeting.
 */

contractorMeetingRoutes.get(
  "/get-contractor-meeting/:id",
  verifyToken,
  sideBarPermissionCheck("view", 271, "contractor_meeting"),
  controllerWrapper(viewAllMeeting)
);

/**
 * @swagger
 * /api/contractorMeeting/delete-contractor-meeting/{id}:
 *   delete:
 *     summary: Delete a specific contractor meeting by ID
 *     tags: [Contractor Meeting]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor meeting
 *     responses:
 *       200:
 *         description: Contractor Meeting deleted successfully.
 */

contractorMeetingRoutes.delete(
  "/delete-contractor-meeting/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 271, "contractor_meeting"),
  controllerWrapper(deleteContractorMeeting)
);

export default contractorMeetingRoutes;
