import { Router } from "express";
import {
  createUpdateContractorRiskAssessment,
  getContractorRiskAssessment,
  deleteContractorRiskAssessment,
} from "../controller/contractorRiskAssessmentController.js";

import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { verifyToken } from "../helper/tokenVerify.js";

/**
 * @swagger
 * tags:
 *   name: Contractor Risk Assessment
 *   description: API operations for managing contractor risk assessments
 */

const contractorRiskAssessmentRoutes = Router();

/**
 * @swagger
 * /api/contractorRiskAssessment/create-contractor-risk-assessment:
 *   post:
 *     summary: Create or update a contractor risk assessment
 *     tags: [Contractor Risk Assessment]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             risk_register: 1
 *             risk_owner: 2
 *             risk_name: "Safety Hazard"
 *             risk_description: "Risk associated with safety hazards during construction."
 *             impact_description: "Potential injuries or fatalities."
 *             causes_description: "Lack of safety protocols."
 *             inherent_likelihood: "High"
 *             inherent_consequence: "Severe"
 *             control_name: "Safety Training"
 *             control_design_intent: "Ensure all workers are trained."
 *             control_effectiveness: "Moderate"
 *             control_owner: 3
 *             control_description: "Regular safety training sessions."
 *             opportunity_identification: 0
 *             priority_confirmation: 1
 *             organization: 5
 *     responses:
 *       200:
 *         description: Contractor risk assessment created or updated successfully.
 */
contractorRiskAssessmentRoutes.post(
  "/create-contractor-risk-assessment",
    verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      255,
      "contractor_risk_assessment"
    )(req, res, next);
  },
  controllerWrapper(createUpdateContractorRiskAssessment)
);

/**
 * @swagger
 * /api/contractorRiskAssessment/get-contractor-risk-assessment:
 *   get:
 *     summary: Get all contractor risk assessments
 *     tags: [Contractor Risk Assessment]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: A list of contractor risk assessments.
 */

contractorRiskAssessmentRoutes.get(
  "/get-contractor-risk-assessment",
  verifyToken,
  sideBarPermissionCheck("view", 255, "contractor_risk_assessment"),
  controllerWrapper(getContractorRiskAssessment)
);

/**
 * @swagger
 * /api/contractorRiskAssessment/get-contractor-risk-assessment/{id}:
 *   get:
 *     summary: Get a specific contractor risk assessment by ID
 *     tags: [Contractor Risk Assessment]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor risk assessment
 *     responses:
 *       200:
 *         description: Details of the contractor risk assessment.
 */

contractorRiskAssessmentRoutes.get(
  "/get-contractor-risk-assessment/:id",
  verifyToken,
  sideBarPermissionCheck("view", 255, "contractor_risk_assessment"),
  controllerWrapper(getContractorRiskAssessment)
);

/**
 * @swagger
 * /api/contractorRiskAssessment/delete-contractor-risk-assessment/{id}:
 *   delete:
 *     summary: Delete a specific contractor risk assessment by ID
 *     tags: [Contractor Risk Assessment]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the contractor risk assessment
 *     responses:
 *       200:
 *         description: Contractor risk assessment deleted successfully.
 */

contractorRiskAssessmentRoutes.delete(
  "/delete-contractor-risk-assessment/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 255, "contractor_risk_assessment"),
  controllerWrapper(deleteContractorRiskAssessment)
);

export default contractorRiskAssessmentRoutes;
