// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdateContractorRiskRegister,
  getContractorRiskRegister,
  deleteContractorRiskRegister,
} from "../controller/contractorRiskRegisterController.js";
import { validateRequest } from "../helper/general.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { strategicRiskValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: contractorRiskRegister
 *   description: API Operations For Managing Contractor Risk Register
 */
const contractorRiskRegisterRoutes = Router();

/**
 * @swagger
 * /api/contractorRiskRegister/create-update-contractor-risk-register:
 *   post:
 *     summary: Create Or Update A tactical Risk
 *     tags: [contractorRiskRegister]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id:id,risk_register_name:risk_register_name,assessment_date:assessment_date,review_date:review_date,risk_context:risk_context,risk_assessment_facilitator:risk_assessment_facilitator,risk_register_approver:risk_register_approver,meeting_id:meeting_id,organization:organization,department:department
 */

contractorRiskRegisterRoutes.post(
  "/create-update-contractor-risk-register",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      293,
      "contractor_risk_register"
    )(req, res, next);
  },
  validateRequest(strategicRiskValidationSchema),
  controllerWrapper(createUpdateContractorRiskRegister)
);

/**
 * @swagger
 * api/contractorRiskRegister/get-contractor-risk-register:
 *   get:
 *     summary: Get list of all Contractor Risk Register List
 *     tags: [contractorRiskRegister]
 */
contractorRiskRegisterRoutes.get(
  "/get-contractor-risk-register",
  verifyToken,
  sideBarPermissionCheck("view", 293, "contractor_risk_register"),
  controllerWrapper(getContractorRiskRegister)
);

/**
 * @swagger
 * api/contractorRiskRegister/get-contractor-risk-register/{id}:
 *   get:
 *     summary: Get A Specific Contractor Risk Register record
 *     tags: [contractorRiskRegister]
 */
contractorRiskRegisterRoutes.get(
  "/get-contractor-risk-register/:id",
  verifyToken,
  sideBarPermissionCheck("view", 293, "contractor_risk_register"),
  controllerWrapper(getContractorRiskRegister)
);

/**
 * @swagger
 * api/contractorRiskRegister/delete-contractor-risk-register/{id}:
 *   delete:
 *     summary: delete a specific record of Contractor Risk Register
 *     tags: [contractorRiskRegister]
 */
contractorRiskRegisterRoutes.delete(
  "/delete-contractor-risk-register/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 293, "contractor_risk_register"),
  controllerWrapper(deleteContractorRiskRegister)
);

export default contractorRiskRegisterRoutes;
