import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { createUpdateDisposalSite, deleteDisposalSite, viewDisposalSite } from "../controller/disposalSiteController.js";

/**
 * @swagger
 * tags:
 *   name:  disposal site
 *   description: API operations for managing  disposal site
 */
const disposalSiteRoutes = Router();

/**
 * @swagger
 * /api/disposalSite/create-update-disposal-site:
 *   post:
 *     summary: Create or update a  disposal site
 *     tags: [ disposal site]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: { id: "id", title: "title", description: "description" }
 */
disposalSiteRoutes.post(
  "/create-update-disposal-site",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 405 , "disposal_site")(req, res, next);
  },
  controllerWrapper(createUpdateDisposalSite)
);

/**
 * @swagger
 * /api/disposalSite/get-disposal-site:
 *   get:
 *     summary: Get all records of  disposal site
 *     tags: [ disposal site]
 */
disposalSiteRoutes.get(
  "/get-disposal-site",
  verifyToken,
  sideBarPermissionCheck("view", 405, "disposal_site"),
  controllerWrapper(viewDisposalSite)
);

/**
 * @swagger
 * /api/disposalSite/get-disposal-site/{id}:
 *   get:
 *     summary: Get a specific record of  disposal site
 *     tags: [ disposal site]
 */
disposalSiteRoutes.get(
  "/get-disposal-site/:id",
  verifyToken,
  sideBarPermissionCheck("view", 405, "disposal_site"),
  controllerWrapper(viewDisposalSite)
);

/**
 * @swagger
 * /api/disposalSite/delete-disposal-site/{id}:
 *   delete:
 *     summary: Delete a specific record of  disposal site
 *     tags: [ disposal site]
 */
disposalSiteRoutes.delete(
  "/delete-disposal-site/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 405, "disposal_site"),
  controllerWrapper(deleteDisposalSite)
);

export default disposalSiteRoutes;