// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  createUpdatedFocusArea,
  deleteFocusArea,
  viewAllFocusArea,
} from "../controller/focusAreaController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { focusAreaCreateUpdateValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: Focus Area
 *   description: API operations for managing all Focus Area
 */
const focusAreaRoutes = Router();

/**
 * @swagger
 * /api/focusArea/create-update-focus-area:
 *   post:
 *     summary: Create or update a Focus Area
 *     tags: [Focus Area]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: integer
 *           example:
 *             id: "id"
 *             name: "name"
 *             description: "description"
 *             organization: [1, 2]
 *     responses:
 *       200:
 *         description: Focus Area successfully created or updated
 */
focusAreaRoutes.post(
  "/create-update-focus-area",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 144, "focus_area")(
      req,
      res,
      next
    );
  },
  validateRequest(focusAreaCreateUpdateValidationSchema),
  controllerWrapper(createUpdatedFocusArea)
);

/**
 * @swagger
 * /api/focusArea/get-focus-area:
 *   get:
 *     summary: Get list of all Focus Area
 *     tags: [Focus Area]
 *     responses:
 *       200:
 *         description: List of all Focus Areas
 */
focusAreaRoutes.get(
  "/get-focus-area",
  verifyToken,
  sideBarPermissionCheck("view", 144, "focus_area"),
  controllerWrapper(viewAllFocusArea)
);

/**
 * @swagger
 * /api/focusArea/get-focus-area/{id}:
 *   get:
 *     summary: Get a specific record of Focus Area
 *     tags: [Focus Area]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the Focus Area
 *     responses:
 *       200:
 *         description: Focus Area data retrieved successfully
 *       404:
 *         description: Focus Area not found
 */
focusAreaRoutes.get(
  "/get-focus-area/:id",
  verifyToken,
  sideBarPermissionCheck("view", 144, "focus_area"),
  controllerWrapper(viewAllFocusArea)
);

/**
 * @swagger
 * /api/focusArea/delete-focus-area/{id}:
 *   delete:
 *     summary: Delete a specific record of Focus Area
 *     tags: [Focus Area]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the Focus Area to delete
 *     responses:
 *       200:
 *         description: Focus Area successfully deleted
 *       404:
 *         description: Focus Area not found
 */
focusAreaRoutes.delete(
  "/delete-focus-area/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 144, "focus_area"),
  controllerWrapper(deleteFocusArea)
);

export default focusAreaRoutes;
