// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import {
  createUpdateLabels,
  deleteLabels,
  viewLabels,
} from "../controller/labelsController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Labels
 *   description: API operations for managing all Labels
 */
const labelsRoutes = Router();

/**
 * @swagger
 * /api/labels/create-update-labels:
 *   post:
 *     summary: Create or update an Labels
 *     tags: [Labels]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: string
 *               name:
 *                 type: string
 *               description:
 *                 type: string
 *               organization:
 *                 type: array
 *                 items:
 *                   type: string
 *           example:
 *             id: "id"
 *             name: "Policy Impact Name"
 *             description: "Description of the insurance policy impact"
 *             organization: ["Organization1", "Organization2"]
 *     responses:
 *       200:
 *         description: Insurance policy impact created or updated successfully
 *       400:
 *         description: Bad request. Invalid input parameters
 *       500:
 *         description: Internal server error
 */
labelsRoutes.post(
  "/create-update-labels",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(req.body.id ? "edit" : "create", 250, "labels")(
      req,
      res,
      next
    );
  },
  controllerWrapper(createUpdateLabels)
);

/**
 * @swagger
 * /api/labels/get-labels:
 *   get:
 *     summary: Get list of all Labels
 *     tags: [Labels]
 *     responses:
 *       200:
 *         description: List of insurance policy impacts
 *       500:
 *         description: Internal server error
 */
labelsRoutes.get(
  "/get-labels",
  verifyToken,
  sideBarPermissionCheck("view", 250, "labels"),
  controllerWrapper(viewLabels)
);

/**
 * @swagger
 * /api/labels/get-labels/{id}:
 *   get:
 *     summary: Get a specific record of Labels
 *     tags: [Labels]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the insurance policy impact
 *     responses:
 *       200:
 *         description: Insurance policy impact data
 *       404:
 *         description: Insurance policy impact not found
 *       500:
 *         description: Internal server error
 */
labelsRoutes.get(
  "/get-labels/:id",
  verifyToken,
  sideBarPermissionCheck("view", 250, "labels"),
  controllerWrapper(viewLabels)
);

/**
 * @swagger
 * /api/labels/delete-labels/{id}:
 *   delete:
 *     summary: Delete a specific record of Labels
 *     tags: [Labels]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: string
 *         required: true
 *         description: The ID of the insurance policy impact to delete
 *     responses:
 *       200:
 *         description: Insurance policy impact successfully deleted
 *       404:
 *         description: Insurance policy impact not found
 *       500:
 *         description: Internal server error
 */
labelsRoutes.delete(
  "/delete-labels/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 250, "labels"),
  controllerWrapper(deleteLabels)
);

export default labelsRoutes;
