// Import necessary modules
import { Router } from "express";

import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  addCommentToMeetingApproval,
  viewAllMeetingApproval,
  viewAllMeetingApprovalById,
} from "../controller/meetingApprovalController.js";
import { controllerWrapper } from "../helper/wrapper.js";

const meetingApprovalRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Meeting Approval
 *   description: API operations for managing all Meeting Approval
 */

/**
 * @swagger
 * /api/meeting-approval/get-meeting-approval:
 *   get:
 *     summary: Get list of all Meeting Approval
 *     tags: [Meeting Approval]
 *     responses:
 *       200:
 *         description: A list of all meeting approvals
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
meetingApprovalRoutes.get(
  "/get-meeting-approval",
  verifyToken,
  sideBarPermissionCheck("view", 26, "meeting_recording"),
  controllerWrapper(viewAllMeetingApproval)
);

/**
 * @swagger
 * /api/meeting-approval/get-meeting-approvalById/{meetingId}:
 *   get:
 *     summary: Get a specific record of Meeting Approval by meetingId
 *     tags: [Meeting Approval]
 *     parameters:
 *       - in: path
 *         name: meetingId
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the meeting
 *     responses:
 *       200:
 *         description: A specific meeting approval record
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
meetingApprovalRoutes.get(
  "/get-meeting-approvalById/:meetingId",
  verifyToken,
  sideBarPermissionCheck("view", 26, "meeting_recording"),
  controllerWrapper(viewAllMeetingApprovalById)
);

/**
 * @swagger
 * /api/meeting-approval/add-comment-to-meeting-approval:
 *   post:
 *     summary: Add comment and approval status to a specific Meeting Approval
 *     tags:
 *       - Meeting Approval
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               id:
 *                 type: number
 *               status:
 *                 type: string
 *               comment:
 *                 type: string
 *           example:
 *             id: 1
 *             status: "Approved"
 *             comment: "Everything looks good!"
 *     responses:
 *       200:
 *         description: Successful operation
 *       400:
 *         description: Bad request
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
meetingApprovalRoutes.post(
  "/add-comment-to-meeting-approval",
  verifyToken,
  sideBarPermissionCheck("edit", 26, "meeting_recording"),
  controllerWrapper(addCommentToMeetingApproval)
);

export default meetingApprovalRoutes;
