// Import necessary modules
import { Router } from "express";
import {
  createUpdateMeetingRecording,
  viewAllMeetingRecording,
  deleteMeetingRecording,
  viewAllMeetingRecordingById,
  getMeetingRecordActionItem,
  meetingNotesStatusChange,
  updateApprovalStatus,
} from "../controller/meetingRecordingController.js";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import { validateRequest } from "../helper/general.js";
import { meetingCreateUpdateRecordingSchema } from "../validationsRecord/validationSchema.js";
import { controllerWrapper } from "../helper/wrapper.js";

/**
 * @swagger
 * tags:
 *   name: Meeting Recording
 *   description: API operations for managing all Meeting Recording
 */
const meetingRecordingRoutes = Router();

/**
 * @swagger
 * /api/meetingRecording/create-update-meeting-recording:
 *   post:
 *     summary: Create or update a Meeting Recording
 *     tags: [Meeting Recording]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: {
 *             "id": 1,
 *             "scheduled_meeting": "Meeting Title",
 *             "actual_meeting_date_from": "2024-01-01T10:00:00Z",
 *             "actual_meeting_date_to": "2024-01-01T11:00:00Z",
 *             "meeting_duration": 60,
 *             "organization": "Org Name",
 *             "department": "Department Name",
 *             "meeting_scriber": "Scriber Name",
 *             "participants": ["Participant 1", "Participant 2"],
 *             "meeting_notes": "Notes for the meeting",
 *             "status": "Scheduled",
 *             "approval_status": "Pending"
 *           }
 */
meetingRecordingRoutes.post(
  "/create-update-meeting-recording",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      25,
      "meeting_recording"
    )(req, res, next);
  },
  validateRequest(meetingCreateUpdateRecordingSchema),
  controllerWrapper(createUpdateMeetingRecording)
);

/**
 * @swagger
 * /api/meetingRecording/get-meeting-recording:
 *   get:
 *     summary: Get all list of Meeting Recording
 *     tags: [Meeting Recording]
 *     responses:
 *       200:
 *         description: A list of all meeting recordings
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
meetingRecordingRoutes.get(
  "/get-meeting-recording",
  verifyToken,
  sideBarPermissionCheck("view", 25, "meeting_recording"),
  controllerWrapper(viewAllMeetingRecording)
);

/**
 * @swagger
 * /api/meetingRecording/get-meeting-recording/{id}:
 *   get:
 *     summary: Get a specific record of Meeting Recording
 *     tags: [Meeting Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the meeting recording
 *     responses:
 *       200:
 *         description: A specific meeting recording record
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
meetingRecordingRoutes.get(
  "/get-meeting-recording/:id",
  verifyToken,
  sideBarPermissionCheck("view", 25, "meeting_recording"),
  controllerWrapper(viewAllMeetingRecording)
);

/**
 * @swagger
 * /api/meetingRecording/get-meeting-recordingById/{meetingId}:
 *   get:
 *     summary: Get a specific record of Meeting Recording by meetingId
 *     tags: [Meeting Recording]
 *     parameters:
 *       - in: path
 *         name: meetingId
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the meeting
 *     responses:
 *       200:
 *         description: A specific meeting recording by meetingId
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
meetingRecordingRoutes.get(
  "/get-meeting-recordingById/:meetingId",
  verifyToken,
  sideBarPermissionCheck("view", 25, "meeting_recording"),
  controllerWrapper(viewAllMeetingRecordingById)
);

/**
 * @swagger
 * /api/meetingRecording/delete-meeting-recording/{id}:
 *   delete:
 *     summary: Delete a specific record of Meeting Recording
 *     tags: [Meeting Recording]
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *         description: The ID of the meeting recording
 *     responses:
 *       200:
 *         description: Successfully deleted
 *       401:
 *         description: Unauthorized
 *       404:
 *         description: Not found
 *       500:
 *         description: Internal server error
 */
meetingRecordingRoutes.delete(
  "/delete-meeting-recording/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 25, "meeting_recording"),
  controllerWrapper(deleteMeetingRecording)
);

/**
 * @swagger
 * /api/meetingRecording/get-meeting-recording-action-item:
 *   get:
 *     summary: Get action items from Meeting Recording
 *     tags: [Meeting Recording]
 *     responses:
 *       200:
 *         description: A list of action items
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
meetingRecordingRoutes.get(
  "/get-meeting-recording-action-item",
  verifyToken,
  sideBarPermissionCheck("view", 25, "meeting_recording"),
  controllerWrapper(getMeetingRecordActionItem)
);

/**
 * @swagger
 * /api/meetingRecording/meeting-notes-status-change:
 *   post:
 *     summary: Change the status of Meeting Notes
 *     tags: [Meeting Recording]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example: {
 *             "id": 1,
 *             "meeting_action_progress": "In Progress",
 *             "status": "Updated"
 *           }
 */
meetingRecordingRoutes.post(
  "/meeting-notes-status-change",
  verifyToken,
  // sideBarPermissionCheck("create", 25, "meeting_recording"),
  controllerWrapper(meetingNotesStatusChange)
);

/**
 * @swagger
 * /api/meetingRecording/update-approval-status:
 *   get:
 *     summary: Update the approval status
 *     tags: [Meeting Recording]
 *     responses:
 *       200:
 *         description: Approval status updated successfully
 *       401:
 *         description: Unauthorized
 *       500:
 *         description: Internal server error
 */
meetingRecordingRoutes.get(
  "/update-approval-status",
  controllerWrapper(updateApprovalStatus)
);

export default meetingRecordingRoutes;
