/**
 * @swagger
 * tags:
 *   name: Non Compliance Technical Form
 *   description: Non-compliance technical forms management
 */

import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";
import {
  createNonComplianceStopCertificateForm,
  getNonComplianceStopCertificateForm,
  deleteNonComplianceStopCertificateForm,
} from "../controller/nonComplianceTechnicalFormController.js";

const nonComplianceTechnicalRoutes = Router();

/**
 * @swagger
 * /api/nonComplianceTechnicalForm/create-non-compliance-technical-form:
 *   post:
 *     summary: Create a non-compliance technical form
 *     tags: [Non Compliance Technical Form]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               organization:
 *                 type: integer
 *               department:
 *                 type: integer
 *               stop_certificate_id:
 *                 type: string
 *               ncr_id:
 *                 type: string
 *               non_conformance_description:
 *                 type: string
 *               requirements_contravened:
 *                 type: string
 *               conditions_of_stop_certificate:
 *                 type: string
 *               recipient_full_name:
 *                 type: string
 *               recipient_role_name:
 *                 type: string
 *               recipient_contact_number:
 *                 type: string
 *               date_received:
 *                 type: string
 *                 format: date
 *               employee_number:
 *                 type: string
 *               acceptance:
 *                 type: string
 *               specify:
 *                 type: string
 *               issuer_full_name:
 *                 type: string
 *               issuer_role_name:
 *                 type: string
 *               issuer_contact_number:
 *                 type: string
 *               issuer_date_received:
 *                 type: string
 *                 format: date
 *               issuer_time:
 *                 type: string
 *                 format: time
 *               has_necessary_conditions:
 *                 type: string
 *                 enum: ["Yes", "No"]
 *               stop_certificate_lifted_required:
 *                 type: string
 *                 enum: ["Yes", "No"]
 *               verifier_full_name:
 *                 type: string
 *               verifier_role_name:
 *                 type: string
 *               verifier_contact_number:
 *                 type: string
 *               verifier_date:
 *                 type: string
 *                 format: date
 *               date_lifted:
 *                 type: string
 *                 format: date
 *               time_lifted:
 *                 type: string
 *                 format: time
 *               compliance_to_mitigation:
 *                 type: string
 *               signature_original_issuer:
 *                 type: string
 *               signature_accepting_lift:
 *                 type: string
 *               deleted:
 *                 type: integer
 *               created_by:
 *                 type: integer
 *               updated_by:
 *                 type: integer
 *             example:
 *               organization: 1
 *               department: 2
 *               incident_id : 1
 *               category_id : 1
 *               stop_certificate_id: "SC-001"
 *               ncr_id: "NCR-123"
 *               non_conformance_description: "Description of non-conformance"
 *               requirements_contravened: "Contravened requirements"
 *               conditions_of_stop_certificate: "Conditions details"
 *               recipient_full_name: "John Doe"
 *               recipient_role_name: "Manager"
 *               recipient_contact_number: "1234567890"
 *               date_received: "2023-07-01"
 *               employee_number: "E001"
 *               acceptance: "Accepted"
 *               specify: "Specify details"
 *               issuer_full_name: "Jane Smith"
 *               issuer_role_name: "Officer"
 *               issuer_contact_number: "0987654321"
 *               issuer_date_received: "2023-07-02"
 *               issuer_time: "10:00:00"
 *               has_necessary_conditions: "Yes"
 *               stop_certificate_lifted_required: "No"
 *               verifier_full_name: "Alice Johnson"
 *               verifier_role_name: "Verifier"
 *               verifier_contact_number: "1122334455"
 *               verifier_date: "2023-07-03"
 *               date_lifted: "2023-07-10"
 *               time_lifted: "15:30:00"
 *               compliance_to_mitigation: "Mitigation details"
 *               signature_original_issuer: "John Doe"
 *               signature_accepting_lift: "Jane Smith"
 *               deleted: 0
 *               created_by: 1
 *               updated_by: 1
 */
nonComplianceTechnicalRoutes.post(
  "/create-non-compliance-technical-form",
  verifyToken,
  controllerWrapper(createNonComplianceStopCertificateForm)
);

/**
 * @swagger
 * /api/nonComplianceTechnicalForm/get-non-compliance-technical-form/{id}:
 *   get:
 *     summary: Get a non-compliance technical form by ID
 *     tags: [Non Compliance Technical Form]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Non-compliance technical form retrieved successfully
 *       404:
 *         description: Form not found
 */
nonComplianceTechnicalRoutes.get(
  "/get-non-compliance-technical-form/:id",
  verifyToken,
  controllerWrapper(getNonComplianceStopCertificateForm)
);

/**
 * @swagger
 * /api/nonComplianceTechnicalForm/delete-non-compliance-technical-form/{id}:
 *   delete:
 *     summary: Delete a non-compliance technical form by ID
 *     tags: [Non Compliance Technical Form]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: id
 *         required: true
 *         schema:
 *           type: integer
 *     responses:
 *       200:
 *         description: Non-compliance technical form deleted successfully
 *       404:
 *         description: Form not found
 */
nonComplianceTechnicalRoutes.delete(
  "/delete-non-compliance-technical-form/:id",
  verifyToken,
  controllerWrapper(deleteNonComplianceStopCertificateForm)
);

export default nonComplianceTechnicalRoutes;
