// Import necessary modules
import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";
import {
  approveRejectTacticalRiskIdentification,
  deleteTacticalRiskIdentification,
  tacticalRiskIdentificationCreateUpdate,
  updateTacticalRiskIdentificationPriority,
  viewTacticalRiskIdentification,
  viewTacticalRiskIdentificationApprovalWorkflow,
} from "../controller/tacticalRiskIdentificationController.js";
import { controllerWrapper } from "../helper/wrapper.js";
import { validateRequest } from "../helper/general.js";
import { tacticalRiskIdentificationValidationSchema } from "../validationsRecord/validationSchema.js";

/**
 * @swagger
 * tags:
 *   name: tacticalRiskIdentificationRoutes
 *   description: API Operations For Managing tactical risk identification
 */
const tacticalRiskIdentificationRoutes = Router();

/**
 * @swagger
 * /api/tacticalRiskIdentificationRoutes/create-update-tactical-risk-identification:
 *   post:
 *     summary: Create Or Update A tactical risk identification
 *     tags: [tacticalRiskIdentificationRoutes]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             id: "id"
 *             risk_register_id: "risk_register_id"
 *             risk_owner: "risk_owner"
 *             risk_name: "risk_name"
 *             risk_description: "risk_description"
 *             impact_description: "impact_description"
 *             causes_description: "causes_description"
 *             inherent_likelihood: "inherent_likelihood"
 *             inherent_consequence: "inherent_consequence"
 *             inherent_risk_rating: "inherent_risk_rating"
 *             inherent_risk_ranking: "inherent_risk_ranking"
 *             control_id: "control_id"
 *             control_name: "control_name"
 *             control_description: "control_description"
 *             control_design_intent: "control_design_intent"
 *             control_effectiveness: "control_effectiveness"
 *             control_owner: "control_owner"
 *             risk_rating: "risk_rating"
 *             risk_ranking: "risk_ranking"
 *             opportunity_identification: "opportunity_identification"
 *             opportunity_description: "opportunity_description"
 *             priority_confirmation: "priority_confirmation"
 *             organization: "organization"
 *             department: "department"
 */
tacticalRiskIdentificationRoutes.post(
  "/create-update-tactical-risk-identification",
  verifyToken,
  (req, res, next) => {
    sideBarPermissionCheck(
      req.body.id ? "edit" : "create",
      44,
      "tactical_risk_identification"
    )(req, res, next);
  },
  validateRequest(tacticalRiskIdentificationValidationSchema),
  controllerWrapper(tacticalRiskIdentificationCreateUpdate)
);

/**
 * @swagger
 * /api/tacticalRiskIdentificationRoutes/get-tactical-risk-identification:
 *   get:
 *     summary: Get list of all tactical risk identification
 *     tags: [tacticalRiskIdentificationRoutes]
 */
tacticalRiskIdentificationRoutes.get(
  "/get-tactical-risk-identification",
  verifyToken,
  sideBarPermissionCheck("view", 44, "tactical_risk_identification"),
  controllerWrapper(viewTacticalRiskIdentification)
);

tacticalRiskIdentificationRoutes.post(
  "/update-tactical-risk-identification-priority",
  verifyToken,
  sideBarPermissionCheck("view", 44, "tactical_risk_identification"),
  controllerWrapper(updateTacticalRiskIdentificationPriority)
);

/**
 * @swagger
 * /api/tacticalRiskIdentificationRoutes/get-tactical-risk-identification:
 *   get:
 *     summary: Get list of all tactical risk identification
 *     tags: [tacticalRiskIdentificationRoutes]
 */
tacticalRiskIdentificationRoutes.get(
  "/get-tactical-risk-identification-approval-workflow",
  verifyToken,
  sideBarPermissionCheck("view", 44, "tactical_risk_identification"),
  controllerWrapper(viewTacticalRiskIdentificationApprovalWorkflow)
);


/**
 * @swagger
 * /api/tacticalRiskIdentificationRoutes/get-tactical-risk-identification/{id}:
 *   get:
 *     summary: Get A Specific tactical risk identification record
 *     tags: [tacticalRiskIdentificationRoutes]
 */
tacticalRiskIdentificationRoutes.get(
  "/get-tactical-risk-identification/:id",
  verifyToken,
  sideBarPermissionCheck("view", 44, "tactical_risk_identification"),
  controllerWrapper(viewTacticalRiskIdentification)
);

/**
 * @swagger
 * /api/tacticalRiskIdentificationRoutes/delete-tactical-risk-identification/{id}:
 *   delete:
 *     summary: Delete a specific record of tactical risk identification
 *     tags: [tacticalRiskIdentificationRoutes]
 */
tacticalRiskIdentificationRoutes.delete(
  "/delete-tactical-risk-identification/:id",
  verifyToken,
  sideBarPermissionCheck("delete", 44, "tactical_risk_identification"),
  controllerWrapper(deleteTacticalRiskIdentification)
);

/**
 * @swagger
 * /api/tacticalRiskIdentificationRoutes/approve-reject-tactical-risk-identification:
 *   post:
 *     summary: Approve or reject tactical risk identification
 *     tags: [tacticalRiskIdentificationRoutes]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           example:
 *             tacticalRiskIdentificationId: "tacticalRiskIdentificationId"
 *             status: "status"
 *             comment: "comment"
 */
tacticalRiskIdentificationRoutes.post(
  "/approve-reject-tactical-risk-identification",
  verifyToken,
  sideBarPermissionCheck("view", 44, "tactical_risk_identification"),
  controllerWrapper(approveRejectTacticalRiskIdentification)
);

export default tacticalRiskIdentificationRoutes;
