import { Router } from "express";
import { verifyToken } from "../helper/tokenVerify.js";
import { controllerWrapper } from "../helper/wrapper.js";

import {
  createUpdateTrainingSchedule,
  getTrainingSchedule,
  deleteTrainingSchedule,
  fetchTrainingScheduleData,
} from "../controller/trainingScheduleController.js";
import { sideBarPermissionCheck } from "../helper/checkPermission.js";

const trainingScheduleRoutes = Router();

/**
 * @swagger
 * tags:
 *   name: Training Schedule
 *   description: API operations for managing training schedules
 */

/**
 * @swagger
 * components:
 *   schemas:
 *     TrainingSchedule:
 *       type: object
 *       properties:
 *         id:
 *           type: integer
 *           description: The auto-generated ID of the training schedule
 *         training_program:
 *           type: integer
 *           description: ID of the training program
 *         employees:
 *           type: string
 *           description: List of employees participating in the training
 *         training_provider:
 *           type: string
 *           enum: ["Internal", "External"]
 *           description: Type of training provider
 *         external_provider:
 *           type: string
 *           description: Name of the external provider
 *         provider_department:
 *           type: string
 *           description: Department of the provider
 *         period:
 *           type: string
 *           description: Period of the training schedule
 *         financial_year:
 *           type: string
 *           description: Financial year of the training
 *         schedule_type:
 *           type: integer
 *           description: Type of schedule (e.g., single day, multiple days)
 *         single_day:
 *           type: string
 *           format: date
 *           description: Date for a single-day training
 *         single_start_time:
 *           type: string
 *           format: time
 *           description: Start time for a single-day training
 *         single_end_time:
 *           type: string
 *           format: time
 *           description: End time for a single-day training
 *         multiple_day:
 *           type: string
 *           format: date
 *           description: Start date for multiple-day training
 *         multiple_end_day:
 *           type: string
 *           format: date
 *           description: End date for multiple-day training
 *         calendar_start_date:
 *           type: string
 *           format: date
 *           description: Start date for calendar-based training
 *         calendar_end_date:
 *           type: string
 *           format: date
 *           description: End date for calendar-based training
 *         schedule_time:
 *           type: string
 *           description: Schedule time details for the training
 *         department:
 *           type: integer
 *           description: Department ID
 *         organization:
 *           type: integer
 *           description: Organization ID
 *         deleted:
 *           type: integer
 *           description: Deletion status of the record
 *         created_at:
 *           type: string
 *           format: date-time
 *           description: The creation date of the training schedule
 *         created_by:
 *           type: integer
 *           description: ID of the creator
 *         updated_at:
 *           type: string
 *           format: date-time
 *           description: The last update date of the training schedule
 *         updated_by:
 *           type: integer
 *           description: ID of the last updater
 */

/**
 * @swagger
 * /api/trainingSchedule/create-update-training-schedule:
 *   post:
 *     summary: Create or update a training schedule
 *     tags: [Training Schedule]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             $ref: '#/components/schemas/TrainingSchedule'
 *           example:
 *             organization: 3
 *             employees: "[12, 10]" # Updated to reflect that it's a string representation of the list
 *             training_program: 45
 *             training_provider: "Internal"
 *             external_provider: ""
 *             provider_department: "HR"
 *             period: "Monthly"
 *             financial_year: "2024"
 *             schedule_type: 1
 *             single_day: ""
 *             single_start_time: ""
 *             single_end_time: ""
 *             multiple_day: "2024-07-23"
 *             multiple_end_day: "2024-07-24"
 *             calendar_start_date: "2024-07-01"
 *             calendar_end_date: "2024-07-31"
 *             schedule_time: "Full Day"
 *             department: 2
 *             deleted: 0
 *             created_by: 1
 *             updated_by: 1
 *     responses:
 *       200:
 *         description: The created or updated training schedule
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/TrainingSchedule'
 */


trainingScheduleRoutes.post(
  "/create-update-training-schedule",
  verifyToken,
  controllerWrapper(createUpdateTrainingSchedule)
);

/**
 * @swagger
 * /api/trainingSchedule/get-training-schedule:
 *   get:
 *     summary: Get all training schedules
 *     tags: [Training Schedule]
 *     responses:
 *       200:
 *         description: A list of training schedules
 *         content:
 *           application/json:
 *             schema:
 *               type: array
 *               items:
 *                 $ref: '#/components/schemas/TrainingSchedule'
 */
trainingScheduleRoutes.get(
  "/get-training-schedule",
  verifyToken,

  controllerWrapper(getTrainingSchedule)
);

/**
 * @swagger
 * /api/trainingSchedule/get-training-schedule/{id}:
 *   get:
 *     summary: Get a training schedule by ID
 *     tags: [Training Schedule]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the training schedule
 *     responses:
 *       200:
 *         description: The requested training schedule
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/TrainingSchedule'
 */
trainingScheduleRoutes.get(
  "/get-training-schedule/:id",
  verifyToken,

  controllerWrapper(getTrainingSchedule)
);

/**
 * @swagger
 * /api/trainingSchedule/delete-training-schedule/{id}:
 *   delete:
 *     summary: Delete a training schedule by ID
 *     tags: [Training Schedule]
 *     parameters:
 *       - in: path
 *         name: id
 *         schema:
 *           type: integer
 *         required: true
 *         description: ID of the training schedule
 *     responses:
 *       200:
 *         description: The training schedule was deleted
 *         content:
 *           application/json:
 *             example:
 *               message: "Training schedule deleted successfully"
 */
trainingScheduleRoutes.delete(
  "/delete-training-schedule/:id",
  verifyToken,

  controllerWrapper(deleteTrainingSchedule)
);

trainingScheduleRoutes.get(
  "/get-training-schedule",
  verifyToken,
  fetchTrainingScheduleData
);
export default trainingScheduleRoutes;
