// InspectionSchema.js
import sequelize from "./sequelize.js"; // Adjust the import path as necessary
import Organization from "./OrganizationSchema.js";
import User from "./UserSchema.js";
import Department from "./DepartmentSchema.js";
import EmployeeType from "./EmployeeTypeSchema.js"; // Assuming an EmployeeType schema exists
import ContractorRegistration from "./ContractorRegistrationSchema.js"; // Assuming a Contractor schema exists
import Location from "./LocationSchema.js"; // Assuming a Location schema exists
import { DataTypes } from "sequelize";
import Currency from "./CurrencySchema.js";

const Inspection = sequelize.define(
  "Inspection",
  {
    id: {
      type: DataTypes.INTEGER,
      autoIncrement: true,
      primaryKey: true,
    },
    inspection_name: {
      type: DataTypes.STRING(250), // Inspection Name
      allowNull: true,
    },
    inspection_type: {
      type: DataTypes.INTEGER, // Dropdown for Inspection Type
      allowNull: true,
    },
    planned_inspection_start_date: {
      type: DataTypes.DATE, // Planned Inspection Start Date
      allowNull: true,
    },
    planned_inspection_end_date: {
      type: DataTypes.DATE, // Planned Inspection End Date
      allowNull: true,
    },
    inspection_source: {
      type: DataTypes.ENUM("Internal", "External"), // Source can be Internal or External
      allowNull: true,
    },
    // Internal Inspector
    internal_inspector_id: {
      type: DataTypes.INTEGER, // Inspector Name fetched from Employee List
      allowNull: true,
      references: {
        model: EmployeeType,
        key: "id",
      },
    },
    // External Inspector
    external_inspector_ids: {
      type: DataTypes.ARRAY(DataTypes.INTEGER), // Multi-select for external inspectors (fetched from contractors)
      allowNull: true,
      references: {
        model: ContractorRegistration,
        key: "id",
      },
    },
    inspection_owner: {
      type: DataTypes.STRING(250), // Inspection Owner
      allowNull: true,
    },
    // Stakeholders Involved
    stakeholders_involved: {
      type: DataTypes.ENUM("Internal", "External"),
      allowNull: true,
    },
    internal_stakeholder_ids: {
      type: DataTypes.TEXT("long"), // Multi-select for internal stakeholders
      allowNull: true,
    },
    external_stakeholder_name: {
      type: DataTypes.STRING(250), // Free Text for External Stakeholder Name
      allowNull: true,
    },
    external_stakeholder_email: {
      type: DataTypes.STRING(250), // Email Input for External Stakeholder
      allowNull: true,
      validate: {
        isEmail: true, // Validate email format
      },
    },
    inspection_location_ids: {
      type: DataTypes.TEXT("long"), // Dropdown for Inspection Locations
      allowNull: true,
    },
    inspection_scope: {
      type: DataTypes.TEXT("long"), // Description Field for Inspection Scope
      allowNull: true,
    },
    critical_business_processes: {
      type: DataTypes.TEXT, // Critical Business Processes
      allowNull: true,
    },

    // Financial Resources
    financial_resources: {
      type: DataTypes.ENUM("Financial Resource Type", "Value", "Currency"),
      allowNull: true,
    },
    financial_resource_type: {
      type: DataTypes.ENUM("Operating Budget", "Capital Budget"), // Enum for Financial Resource Type
      allowNull: true,
    },
    value: {
      type: DataTypes.FLOAT, // Numerical Input for Financial Value
      allowNull: true,
      validate: {
        isFloat: true, // Validate float input
        min: 0, // Minimum value constraint
      },
    },
    currency_id: {
      type: DataTypes.INTEGER, // Dropdown for Currency, referencing prefilled dataset
      allowNull: true,
      references: {
        model: Currency,
        key: "id",
      },
    },
    evaluation_tool: {
      type: DataTypes.ENUM("Applicable", "Not Applicable"), // Dropdown for Evaluation Tool
      allowNull: true,
    },
    organization: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Organization,
        key: "id",
      },
    },
    created_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    created_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    deleted: {
      type: DataTypes.TINYINT,
      allowNull: true,
      defaultValue: 0,
    },
    updated_at: {
      type: DataTypes.DATE,
      allowNull: true,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
      onUpdate: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    updated_by: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: User,
        key: "id",
      },
    },
    department: {
      type: DataTypes.INTEGER,
      allowNull: true,
      references: {
        model: Department,
        key: "id",
      },
    },
  },
  {
    tableName: "inspections",
    timestamps: false,
    charset: "utf8mb4",
    collate: "utf8mb4_general_ci",
  }
);

// Associations
Inspection.belongsTo(EmployeeType, {
  as: "internalInspector",
  foreignKey: "internal_inspector_id",
});
Inspection.belongsToMany(ContractorRegistration, {
  as: "externalInspectors",
  through: "InspectionExternalInspectors",
  foreignKey: "inspection_id",
});
Inspection.belongsToMany(EmployeeType, {
  as: "internalStakeholders",
  through: "InspectionInternalStakeholders",
  foreignKey: "inspection_id",
});
Inspection.belongsToMany(Location, {
  as: "inspectionLocations",
  through: "InspectionLocationMapping",
  foreignKey: "inspection_id",
});
Inspection.belongsTo(Currency, { as: "currency", foreignKey: "currency_id" });
Inspection.belongsTo(Organization, { as: "org", foreignKey: "organization" });
Inspection.belongsTo(User, { as: "creator", foreignKey: "created_by" });
Inspection.belongsTo(User, { as: "updater", foreignKey: "updated_by" });
Inspection.belongsTo(Department, { as: "dep", foreignKey: "department" });
// console.log(await Inspection.sync({ alter: true }));
export default Inspection;
