import { DataTypes } from "sequelize";
import sequelize from "./sequelize.js";
import Department from "./DepartmentSchema.js";
import User from "./UserSchema.js";
import Organization from "./OrganizationSchema.js";

const interactionFrequenciesArr = [
  "Daily",
  "Weekly",
  "Bi-weekly",
  "Monthly",
  "Bi-Monthly",
  "Quarterly",
  "Bi-Annually",
  "Annually",
];

const levelsArr = ["High", "Medium", "Low"];
const stakeholderTypeArr = ["Internal", "External"];
const classificationArr = ["Primary", "Secondary"];
const connectionStrengthArr = ["Strong", "Moderate", "Weak"];
const classificationStatusArr = ["Active", "Inactive", "Potential"];

const engagementLevelArr = [
  "Provide access to general channels of information and feedback",
  "Regularly update and consult to prevent escalation of concerns",
  "Monitor closely to prevent the development of threats",
  "Ensure high levels of engagement to avoid potential issues",
  "Engage proactively to leverage opportunities and mitigate threats",
];

const StakeHolderClassification = sequelize.define(
  "StakeHolderClassification",
  {
    id: {
      type: DataTypes.INTEGER,
      autoIncrement: true,
      primaryKey: true,
    },
    stakeholder_category: {
      type: DataTypes.STRING(255),
    },
    stakeholder_type: {
      type: DataTypes.ENUM(stakeholderTypeArr),
      allowNull: true,
      defaultValue: "Internal",
    },
    classification: {
      type: DataTypes.ENUM(classificationArr),
      allowNull: true,
      defaultValue: "Primary",
    },
    engagement_level: {
      type: DataTypes.ENUM(levelsArr),
    },
    status: {
      type: DataTypes.ENUM(classificationStatusArr),
    },
    interaction_frequency: {
      type: DataTypes.ENUM(interactionFrequenciesArr),
    },
    influence_level: {
      type: DataTypes.ENUM(levelsArr),
    },
    connection_strength: {
      type: DataTypes.ENUM(connectionStrengthArr),
    },
    interest_score: {
      type: DataTypes.INTEGER,
      validate: {
        max: 9,
        min: 1,
      },
    },
    influence_score: {
      type: DataTypes.INTEGER,
      validate: {
        max: 10,
        min: 1,
      },
    },
    assessment_date: {
      type: DataTypes.DATEONLY,
      allowNull: true,
    },
    aggregate_score: {
      type: DataTypes.STRING(255),
      allowNull: true,
    },
    priority_level: {
      type: DataTypes.ENUM(levelsArr),
    },
    classification_type: {
      type: DataTypes.STRING(255),
    },
    recommended_engagement_level: {
      type: DataTypes.STRING(255),
    },
    unique_id: {
      type: DataTypes.STRING(255),
      allowNull: true,
    },
    organization: {
      type: DataTypes.INTEGER,
      references: {
        model: Organization,
        key: "id",
      },
    },
    department: {
      type: DataTypes.INTEGER,
      references: {
        model: Department,
        key: "id",
      },
    },
    deleted: {
      type: DataTypes.INTEGER,
      allowNull: true,
      defaultValue: 0,
    },
    created_at: {
      type: DataTypes.DATE,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    created_by: {
      type: DataTypes.INTEGER,
      references: {
        model: User,
        key: "id",
      },
    },
    updated_at: {
      type: DataTypes.DATE,
      defaultValue: sequelize.literal("CURRENT_TIMESTAMP"),
      onUpdate: sequelize.literal("CURRENT_TIMESTAMP"),
    },
    updated_by: {
      type: DataTypes.INTEGER,
      references: {
        model: User,
        key: "id",
      },
    },
  },
  {
    timestamps: false,

    tableName: "stakeholder_classification",
    charset: "utf8mb4",
    collate: "utf8mb4_general_ci",
  }
);

StakeHolderClassification.belongsTo(Organization, {
  foreignKey: "organization",
  as: "Organization",
});

StakeHolderClassification.belongsTo(Department, {
  foreignKey: "department",
  as: "ExternalDepartment",
});

StakeHolderClassification.belongsTo(User, {
  foreignKey: "created_by",
  as: "CreatedBy",
});

StakeHolderClassification.belongsTo(User, {
  foreignKey: "updated_by",
  as: "UpdatedBy",
});

// console.log(await StakeHolderClassification.sync({ alter: true }));

export default StakeHolderClassification;
